#include "GPSReference.h"

// pin numbers
#define GPS_TX_PIN      7
#define GPS_RX_PIN      6


/* instantiate a GPS receiver object using the given transmit and receiver Arduino pins.
 */
GPS::GPS()
{
    // set up comm
    gps_serial = new SoftwareSerial (GPS_TX_PIN, GPS_RX_PIN);
    gps_serial->begin(9600);

    // init
    capture_i = 0;
    parsed_i = 0;
    gps_line_len = 0;
    worst_snr = best_snr = 0; // match dpy.h prev_b/wsnr
    memset (prns, 0, sizeof(prns));
    memset (gps_lines, 0, sizeof(gps_lines));
    // TODO reset();
    delay(1000);

    // select GPRMC every 1, GPGSV and GPGSA every 5
    // hint: compute checksum at http://www.hhhh.org/wiml/proj/nmeaxor.html
    gps_serial->println("$PMTK314,0,1,0,0,5,5,0,0,0,0,0,0,0,0,0,0,0,0,0*29");

    // Timer0 is already used for millis() - we'll just interrupt somewhere
    // in the middle to call our GPS support function too
    OCR0A = 0xAF;
    TIMSK0 |= _BV(OCIE0A);

    // let timer run a few times, else handler can get called with partial gps object
    delay(2);
}

// Interrupt called once a millisecond to poll gps
// N.B. beware first few calls before gps set
ISR(TIMER0_COMPA_vect)
{
    if (gps)
	gps->poll();
}


/* parse a NMEA message when ready.
 */
void GPS::parse()
{
    // parse next line until catch up with current line being captured
    if (parsed_i == capture_i)
	return;
    char *line = &gps_lines[parsed_i][0];
    if (++parsed_i == N_GPS_LINES)
	parsed_i = 0;

    // validate
    if (line[0] != '$')
	return;
    uint8_t len;
    uint8_t chksum = 0;
    for (len = 1; len < (MAX_GPS_LINE-3) && line[len] != '*'; len++)	// $ .. *
	chksum ^= line[len];
    if (chksum != 16*parseHex(line[len+1]) + parseHex(line[len+2]))	// * xx
	return;
    // Serial.println (line);		// N.B. this can take so long we can miss some GPS messages

    // parse a known-good NMEA response, moving line along token-by-token
    char *token;
    SatPos sat;
    NowLoc nl;
    uint8_t gsv_prn = 0;
    bool is_gprmc = strncmp (line, "$GPRMC", 6) == 0;
    bool is_gpgsv = strncmp (line, "$GPGSV", 6) == 0;
    bool is_gpgsa = strncmp (line, "$GPGSA", 6) == 0;
    bool gprmc_valid = false;

    // process each token
    for (uint8_t tok_i = 0; (token = strsep (&line, ",*")) != NULL; tok_i++) {

	if (is_gpgsa) {

	    // fields 3-14 contain PRNs of sats used in solution
	    if (tok_i == 1) {
		memset (prns, 0, sizeof(prns));	// init prns list
		// GPGSVs likely next so report and reset best snr
		dpy->drawSNRs (worst_snr, best_snr);
		best_snr = 0;
		worst_snr = ~0;
	    } else if (GSA_MIN <= tok_i && tok_i <= GSA_MAX) {
		prns[tok_i-GSA_MIN] = atoi(token);
		// Serial.print("GSA "); Serial.println(prns[tok_i-GSA_MIN]);
	    }

	} else if (is_gpgsv) {

	    switch (tok_i) {
	    case 0:	// message
	        break;
	    case 1:	// N messages
		break;
	    case 2:	// message number
		break;
	    case 3:	// tot sats in view
		break;
	    default: {
		    // remaining tokens are in groups of 4 per satellite
		    switch (tok_i%4) {
		    case 0:		// ID
			gsv_prn = atoi(token);
			// Serial.print("GSV "); Serial.println(gsv_prn);
			break;
		    case 1:
			sat.el = atoi(token);
			break;
		    case 2:
			sat.az = atoi(token);
			break;
		    case 3:		// snr, missing or 0 means invalid
			uint8_t snr = atoi (token);
			if (snr > 0) {
			    // draw only if this sat is used in solution
			    for (uint8_t i = 0; i < MAX_PRNS; i++) {
				if (prns[i] == gsv_prn) {
				    dpy->drawSat (sat);
				    if (snr > best_snr)
					best_snr = snr;
				    if (snr < worst_snr)
					worst_snr = snr;
				    break;
				}
			    }
			}
			break;
		    }
		}
		break;
	    }

	} else if (is_gprmc) {

	    switch (tok_i) {
	    case 1:	// HHMMSS
		nl.second = atoi (token+4);
		token[4] = '\0';
		nl.minute = atoi (token+2);
		token[2] = '\0';
		nl.hour = atoi (token);
		break;
	    case 2:	// A = OK
		gprmc_valid = token[0] == 'A';
		break;
	    case 3:	// LTMM.MM
		nl.lat = atof(token+2)/60.0F;
		token[2] = '\0';
		nl.lat += atoi(token);
		break;
	    case 4:	// lat N/S
		nl.lat_n = token[0] == 'N';
		break;
	    case 5:	// LNGMM.MM
		nl.lng = atof(token+3)/60.0F;
		token[3] = '\0';
		nl.lng += atoi(token);
		break;
	    case 6:	// lng E/W
		nl.lng_e = token[0] == 'E';
		break;
	    case 9:	// DDMMYY
		nl.year = atoi (token+4);
		token[4] = '\0';
		nl.month = atoi (token+2);
		token[2] = '\0';
		nl.day = atoi (token);
		if (gprmc_valid)
		    dpy->drawNowLoc(nl);
		break;
	    }
	}
    }
}

/* do a full cold reset
 */
void GPS::reset()
{
    gps_serial->println("$PMTK104*37");
}

/* given an ASCII character '0'..'F' return the value as a hex integer value.
 * N.B. we do no bounds checking.
 */
uint8_t GPS::parseHex (const char h)
{
    if (h <= '9')
	return (h - '0');
    return (h - 'A' + 10);
}

/* check for more characters from the receiver, collect into gps_lines[capture_i].
 * N.B. must call this often to avoid dropping characters.
 */
void GPS::poll()
{
    char c;
    while ((c = gps_serial->read()) > 0) {
	if (c == '\n') {
	    gps_lines[capture_i][gps_line_len] = '\0';
	    if (++capture_i == N_GPS_LINES)
		capture_i = 0;
	    gps_line_len = 0;
	} else if (c != '\r' && gps_line_len < MAX_GPS_LINE-1)
	    gps_lines[capture_i][gps_line_len++] = c;
    }
}
