/* 
	afxmit.c - transmit data from PortAudio to network.  Use with afrecv.c.
	version: U1 8/29/2012
	Copyright (c) 2012 Martin S Ewing

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

	Contact author: ewing@alum.mit.edu, 
		28 Wood Rd, Branford, CT, USA 06405

	Portions of code incorporated from examples in PortAudio and Speex documentation:
		www.portaudio.com
		www.speex.org
*/

/* TO DO:
 * Try more codecs
 */
 
#include "af.h"

/* Globals */

struct sockaddr_in server_addr, client_addr;
int	net_sock=0;
int	run_flag=0;		/* = sending data to network */
int	ncodec;
int	codec_chunk_size;	/* samples */
int	net_packets_per_chunk;
int	spx_quality;		// typ. 8
int	spx_complexity;		// typ. 3
int	spx_buffering;		// typ. 4
void 	*spx_state;
SpeexBits 	spx_bits;
int	spx_first = TRUE;

/*  Audio sampling & network transmission scheme:
 *
 *  1. Sampling: 16 bit mono, 8 kHz
 *  2. Header precedes each audio block with byte count to follow, etc.
 *  3. Null codec: 256 samples (512 B) sent per 32 msec.
 *  4. Speex codec: input block 160 B, 4x buffered (640 B = 80 msec)
 *     output block is much smaller.
 *  5. afxmit sends at rate determined by soundcard input clock.
 *     No compensation for fast or slow clock.
 *  (6. Using TCP for predictability (at a price), also allows ssh port forwarding
 *     if needed.)
 *  7. UDP version allows for framing and control of bit slip, less bandwidth.
 */

static int sendCallback( const void *inputBuffer, void *outputBuffer,
        		unsigned long framesPerBuffer,
        		const PaStreamCallbackTimeInfo* timeInfo,
			PaStreamCallbackFlags statusFlags,
			void *userData )
{
	const SAMPLE 	*rptr = (const SAMPLE*)inputBuffer;
	int 		b_size;
	static char	net_buf[NET_BUF_SIZE];
	static int	net_buf_count;
	
	/* Speex related */
	char		SpeexOut[NET_BUF_SIZE];
	int		nbBytes;
	static void	*nxt;
	int		availbuf;
	char    	s_nbBytes[8];
	int		fsize;

	extern int	spx_buffering, spx_quality, spx_complexity;
	extern void 	*spx_state;
	extern SpeexBits spx_bits;
	extern int	spx_first;
	
	(void) outputBuffer; /* Prevent unused variable warnings. */
	(void) timeInfo;
	(void) statusFlags;
	(void) userData;

	assert(framesPerBuffer == codec_chunk_size);
	
	if ( !run_flag ) return paContinue;
	
	switch (ncodec) 
	{
		case CODEC_NULL:
			b_size = sizeof(SAMPLE) * framesPerBuffer;
			sendto(net_sock, rptr, b_size, 0, (const struct sockaddr *)&client_addr, 
				sizeof(client_addr));
			break;
		case CODEC_SPEEX:
			if (spx_first) 
			{
				speex_bits_init(&spx_bits);
				spx_state = speex_encoder_init(&speex_nb_mode);
				speex_encoder_ctl(spx_state, SPEEX_GET_FRAME_SIZE, &fsize);
				assert(fsize == 160);
				speex_encoder_ctl(spx_state, SPEEX_SET_QUALITY, &spx_quality);
				speex_encoder_ctl(spx_state, SPEEX_SET_COMPLEXITY,
						&spx_complexity);
				memset(net_buf, 0, NET_BUF_SIZE);
				net_buf_count = 0;
				nxt = net_buf;		/* pointer for xfer */
				availbuf = NET_BUF_SIZE;
				spx_first = FALSE;
			}
			speex_bits_reset(&spx_bits);
			speex_encode_int(spx_state, (spx_int16_t *)rptr, &spx_bits);
			/* Speex output size will depend on requested quality, etc. */
			nbBytes = speex_bits_write(&spx_bits, SpeexOut, NET_BUF_SIZE); // typ 38 b
			availbuf -= (nbBytes + 4);
			assert(availbuf >= 0);
			sprintf(s_nbBytes,"%04d",nbBytes);

			/* First output size of chunk - 4 chars, then 160 B chunk */
			/* May not be necessary, since receiver can figure it out beforehand */
			nxt = memcpy(nxt, s_nbBytes, 4) + 4;
			nxt = memcpy(nxt, SpeexOut, nbBytes) + nbBytes;
			net_buf_count++;
			if (net_buf_count >= spx_buffering) 
			{
				b_size = (int) (nxt - (void *)net_buf);
				sendto(net_sock, net_buf, b_size, 0, 
					(const struct sockaddr *)&client_addr,sizeof(client_addr));
				net_buf_count = 0;
				availbuf = NET_BUF_SIZE;
				nxt = net_buf;
			}
			break;
		default:
			perror("Callback codec unknown");
			return paAbort;
	}
	return paContinue;
}

/******REAL ENTRY**********************************************/

int main(int argc, char *argv[])
{
	PaStreamParameters	inputParameters;
	PaStream	*stream;
	PaError		err = paNoError;
	int		opt, nparm;
	int		port = PORT_DEFAULT; 
	int		codec_parm[6];
	extern int 	optind, opterr, optopt;
	socklen_t	client_addrlen;
	char		cmd_buf[L_CMD];		/* commands from client */
	ssize_t		cmd_len;
	char		*tok, *delim=" ";
	extern void 	*spx_state;
	extern SpeexBits 	spx_bits;
	extern int	run_flag;

	/* afxmit [-p port] */
	
	while ((opt = getopt(argc, argv, "p:")) != -1) 
	{
		switch(opt) 
		{
			case 'p':	/* port */
				port = atoi(optarg);
				break;
			default:	/* error */
				fprintf(stderr, 
				"usage: %s -p port\n", argv[0]);
				exit(EXIT_FAILURE);
		}
	}

	/* Initialize network for UDP command reception */

	if ((net_sock = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
		perr("Socket");
	server_addr.sin_family = AF_INET;
	server_addr.sin_port = htons(port);
	server_addr.sin_addr.s_addr = htonl(INADDR_ANY);	/* restrict at later time? */
	if (bind(net_sock, (const struct sockaddr *)&server_addr, sizeof(server_addr)) < 0)
		perr("Bind");

	/* Wait for request packet from client on control port.
	 * It can be 'CODEC' + codec ID + parms, 'START', 'STOP' (to stop audio packets), 
	 * or 'QUIT' (exit server).
	 */
	printf("Entering recvfrom loop...\n");
	while(TRUE)			// recvfrom loop
	{
		/* wait for, then process a command from the client */
		client_addrlen = sizeof(client_addr);
		cmd_len = recvfrom(net_sock, cmd_buf, L_CMD, 0, 
			(struct sockaddr *)&client_addr, &client_addrlen);
		if (cmd_len < 0) perr("Read cmd");
		cmd_buf[cmd_len] = '\0';		// ensure terminated string
		printf("Command: .%s.\n", cmd_buf);
		if (strncmp(cmd_buf, CMD_CODEC, L_CMD_OP) == 0) //allow for parameters
		{
			if (run_flag)
				printf("WARN: Received Codec command, but we are already running. Ignore.\n");
			else {
				/* CODEC <parm1=codec_no.> <parm2> ...<parm4> */
				tok = strtok(cmd_buf, delim);	// skip codec command name
				nparm = 0;
				while ((tok = strtok(NULL, delim)) != 0)
					codec_parm[nparm++] = atoi(tok);
				ncodec = codec_parm[0];
				/* prepare chosen codec */
				switch(ncodec) 
				{
					case CODEC_NULL:	/* null codec */
					/* CODEC 0 */
						codec_chunk_size = 	256;
						break;
					case CODEC_SPEEX:	/* speex */
					/* CODEC 1 <buffering 1:?> <quality 0:9> <complexity 1:10> */
						assert(nparm > 3);
						codec_chunk_size = 	160;
						spx_buffering =		codec_parm[1];
						spx_quality = 		codec_parm[2];
						spx_complexity = 	codec_parm[3];
						break;
					default:		/* unknown */
						printf("Unknown codec %d\n", ncodec);
						exit(EXIT_FAILURE);
				}
			}
		}
		if (strncmp(cmd_buf, CMD_START, L_CMD_OP) == 0) 
		{
			if (run_flag)
				printf("WARN: Asked to start, but we are already running. Ignore.\n");
			else {
				/* out_addr will be the reply address for audio packets */
				printf("Starting with chunk %d\n", codec_chunk_size);
				if (ncodec == CODEC_SPEEX) 
					printf("buf %d, qual %d, cplx %d\n", 
					spx_buffering, spx_quality, spx_complexity);

				/* Set up portaudio input channel */
				err = Pa_Initialize();
				if( err != paNoError )
					goto done;
				inputParameters.device = Pa_GetDefaultInputDevice(); // default device
				if (inputParameters.device == paNoDevice) 
				{
					fprintf(stderr,"Error: No default input device.\n");
					goto done;
				}
				inputParameters.channelCount = NUM_CHANNELS;	// mono input
				inputParameters.sampleFormat = PA_SAMPLE_TYPE;
				inputParameters.suggestedLatency
					= Pa_GetDeviceInfo( inputParameters.device )
						-> defaultLowInputLatency;
				inputParameters.hostApiSpecificStreamInfo = NULL;

				err = Pa_OpenStream(
					&stream,
					&inputParameters,	// input only
					NULL,
					SAMPLE_RATE,
					codec_chunk_size,
					paClipOff,
					sendCallback,
					NULL );
				if( err != paNoError ) 
					goto done;
				err = Pa_StartStream( stream );
				if( err != paNoError ) 
					goto done;
				run_flag = TRUE;
				printf("=== Sending. ===\n");
			}
		}
		if (strncmp(cmd_buf, CMD_STOP, L_CMD_OP) == 0) // Stop xmsn, await more commands
		{
			Pa_AbortStream(stream);  // not "Stop"
			Pa_CloseStream(stream);
			Pa_Terminate();
			if (!spx_first)	// i.e., if we were running speex
			{
				speex_bits_destroy(&spx_bits);	// destroy speex objects
				speex_encoder_destroy(spx_state);
				spx_first = TRUE;		// prepare to initialize
			}
			run_flag = FALSE;
			printf("=== Pause sending. ===\n");
		}
		if (strncmp(cmd_buf, CMD_QUIT, L_CMD_OP) == 0) // Quit program
		{
			Pa_AbortStream(stream);  // not "Stop"
			Pa_CloseStream(stream);
			Pa_Terminate();
			printf("=== Quitting ===\n");
			exit(EXIT_SUCCESS);
		}
	}	// recvfrom loop

done:	close(net_sock);
	Pa_Terminate();
	if( err != paNoError )
	{
		fprintf( stderr, "An error occured while using the portaudio stream\n" );
		fprintf( stderr, "Error number: %d\n", err );
		fprintf( stderr, "Error message: %s\n", Pa_GetErrorText( err ) );
		err = 1;          /* Always return 0 or 1, but no other return codes. */
	}
	return err;
}
