/******************************************/
/*                                        */
/* STEST: DDS synth test bed              */
/*                                        */
/* Compile with MSVC for Win32            */
/*                                        */
/* ke5fx@qsl.net                          */
/*                                        */
/******************************************/

#include <math.h>
#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <direct.h>
#include <limits.h>
#include <assert.h>
#include <time.h>

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <mmsystem.h>

#define VERSION "1.10"

//
// Include platform-independent typedefs and synthesizer driver class
//

#include "typedefs.h"
#include "synth.cpp"

//
// TRUE to make Tek 49x analyzer follow the output signal
// (requires NatInst GPIB card)
//

#define ANALYZER_CONTROL 0

//
// TRUE to wait for PLL lock line to go high after sending frequency command
//

#define WAIT_FOR_LOCK 1

//
// Synthesizer output range and power-up default
//
// When picking random frequencies, stay in Tek 49x band 1 to save 
// wear and tear on hardware
//

#define FOUT_MIN        970000000
#define FOUT_MAX        2150000000 
#define FOUT_RAND_MAX   1799999999  
#define FOUT_DEFAULT    1000000000

//
// DDS equates
//

#define DDS_CLOCK_FREQ  10000000
#define DDS_CENTER_FREQ 10700000

#define DDS_MIN_MULT    8
#define DDS_MAX_MULT    12

//
// Globals
//

SYNTH *synth = NULL;    // Synthesizer object under test

U32 out_port;           // LPT port base address
S32 SA_update;          // TRUE to follow output with spectrum analyzer 
S32 spur_search_mode;   // TRUE if searching for spurs

/******************************************/
#if ANALYZER_CONTROL
#include "gpiblib.h"

void WINAPI GPIB_error(C8 *msg, S32 ibsta, S32 iberr, S32 ibcntl)
{
   printf("\n\n%s",msg);

   exit(1);
}

void GPIB_update(S32 cur_freq)
{
   GPIB_startup(3,
                GPIB_error);

   C8 buffer[80];
   sprintf(buffer,"FREQ %u",cur_freq);

   GPIB_write(buffer);

   GPIB_shutdown();
}

void GPIB_plot(S32 cur_freq)
{
   GPIB_startup(3,
                GPIB_error);

   GPIB_write("SIGSWP");

   C8 buffer[80];
   sprintf(buffer,"FREQ %u",cur_freq);
   GPIB_write(buffer);

   GPIB_write("SIGSWP");
   GPIB_write("WAIT");

   GPIB_write("PTYPE HP7470");      // select HP7470 plotter compatibility
   GPIB_write("GRAT ON");           // turn on graticule illumination

   sprintf(buffer,"%u (%d).plt",cur_freq,synth->current_CM);

   FILE *out = fopen(buffer,"wt");

   fprintf(out, "%s\n", 
      GPIB_query("plot?"));         // send result of PLOT? to stdout

   fclose(out);

   GPIB_write("TRIG FRERUN");
   GPIB_shutdown();
}

void shutdown(void)
{
   GPIB_shutdown();
}

#endif

/******************************************/
void main(S32 argc, C8 *argv[])
{                	
   U32 i;
   U32 cur_freq,last_freq;
   U32 freq;

   //
   // Display signon banner
   //

   printf("\n");
   printf("Win32 test bed for DDS/PLL hybrid synthesizer\n");
   printf("V"VERSION" of "__DATE__" by John Miles (ke5fx@qsl.net)\n");
   printf("\n\n");

   //
   // Usage: stest port# (1, 2, or 3)
   //

   if (argc == 1)
      {
      printf("Usage: stest <LPT port #>\n");
      exit(1);
      }

   S32 LPT_port = argv[1][0] - '1';

   //
   // Get base addresses of parallel ports
   //
   // In Win2K or XP, this would probably have to be hard-wired to
   // the machine's port address
   // 

   U32 base_address[3];

   _asm
      {
      push es
      mov eax,40h
      mov es,ax
      xor eax,eax
      mov ax,es:[08h]
      mov base_address,eax
      mov ax,es:[0ah]
      mov base_address+4,eax
      mov ax,es:[0ch]
      mov base_address+8,eax
      pop es
      }

   out_port = base_address[LPT_port];

   printf("PLL using LPT%d port at 0x%X\n",LPT_port+1,out_port);

#if ANALYZER_CONTROL
   atexit(shutdown);
#endif

   //
   // Set synthesizer output frequency to 1.0 GHz
   //

   cur_freq          = FOUT_DEFAULT;
   last_freq         = -1;
   SA_update         = 1;
   spur_search_mode  = 0;

   srand(time(NULL));

   //
   // Send data to port
   //

   printf("\n");
   printf("   R: Set random frequency\n");
   printf("   E: Enter frequency in Hz (3- or 4-digit value treated as MHz)\n");
   printf(" 0-8: Decrement frequency by 10^n (Shift to increment)\n");
   printf(" A/Z: Frequency lower/upper bounds\n");
#if ANALYZER_CONTROL
   printf(" u/U: Turn Tek 49x spectrum analyzer updates off/on\n");
   printf("   P: Save Tek 49x plot file for current frequency\n");
   printf("  ^A: Automated random spur search\n");
#endif
   printf(" Spc: Force reload of PLL and DDS registers\n");
   printf("\n");
   printf(" ESC: Quit\n\n");

   while (1)
      {
      Sleep(10);

      rand();

      //
      // Create synthesizer object if it does not already exist
      //

      if (synth == NULL)
         {
         synth = new SYNTH(out_port,
#ifdef PLLATINUM
                           LMX23X6,
#else
                           ADF411X,
#endif
                           FOUT_MIN,
                           DDS_CENTER_FREQ,
                           DDS_CLOCK_FREQ,
                           DDS_MIN_MULT,
                           DDS_MAX_MULT);
         last_freq = -1;
         }

      //
      // Update frequency
      // 

      if (last_freq != cur_freq)
         {
         last_freq = cur_freq;

         printf("Frequency: %lu Hz",cur_freq);

         synth->set_frequency(cur_freq);

         //
         // See how long it takes the PLL to lock
         //

#if WAIT_FOR_LOCK

         S32 start_acquisition = timeGetTime();

         S32 poll = 0;

         while (1)
            {
            if (synth->locked())
               {
               break;
               }

            poll++;

            if (!(poll % 100000))
               {
               printf(".");

               if (kbhit())
                  {
                  printf("Aborted");
                  getch();
                  break;
                  }
               }
            }

         S32 lock_time = timeGetTime() - start_acquisition;
         printf(": Lock time = %d ms", lock_time);
#endif

         printf("\n");

         //
         // Tune 494P to new center frequency
         //

#if ANALYZER_CONTROL
         if (SA_update)
            {
            //
            // If in automated spur-search mode, save a plot for the new 
            // frequency ... otherwise, just set the frequency
            //

            if (spur_search_mode)
               {
               GPIB_plot(cur_freq);
               }
            else
               {
               GPIB_update(cur_freq);
               }
            }
#endif
         }

      //
      // If in automated spur search mode, generate a new random frequency
      // below 1.79 GHz (don't want to wear out the band-switching relay 
      // in the 494!)
      //

      if (spur_search_mode)
         {
         while (1)
            {
            freq = rand();
            freq = (freq << 16) + rand();
            freq %= (FOUT_MAX-FOUT_MIN);
            freq += FOUT_MIN;

            if (freq < FOUT_RAND_MAX)
               {
               break;
               }
            }

         cur_freq = freq;
         }

      //
      // Check for keyboard command
      //

      if (kbhit())
         {
         switch (getch())
            {
            //
            // r: Random frequency
            //

            case 'r':
               freq = rand();
               freq = (freq << 16) + rand();
               freq %= (FOUT_RAND_MAX-FOUT_MIN);
               freq += FOUT_MIN;

               cur_freq = freq;
               break;

            //
            // e: Enter frequency
            //

            case 'e':

               printf("\nEnter new frequency in Hz: ");

               getch();    // weird bug in gets/scanf ungets the last character?
               C8 buff[128];
               gets(buff);

               cur_freq = atol(buff);

               if (cur_freq < 9999)
                  {
                  cur_freq *= 1000000;
                  }

               break;

            //
            // 0-8: Adjust frequency by 10.sup.n
            //

            case '0':
               cur_freq -= 1;
               break;
            case ')':
               cur_freq += 1;
               break;

            case '1':
               cur_freq -= 10;
               break;
            case '!':
               cur_freq += 10;
               break;

            case '2':
               cur_freq -= 100;
               break;
            case '@':
               cur_freq += 100;
               break;

            case '3':
               cur_freq -= 1000;
               break;
            case '#':
               cur_freq += 1000;
               break;

            case '4':
               cur_freq -= 10000;
               break;
            case '$':
               cur_freq += 10000;
               break;

            case '5':
               cur_freq -= 100000;
               break;
            case '%':
               cur_freq += 100000;
               break;

            case '6':
               cur_freq -= 1000000;
               break;
            case '^':
               cur_freq += 1000000;
               break;

            case '7':
               cur_freq -= 10000000;
               break;
            case '&':
               cur_freq += 10000000;
               break;

            case '8':
               cur_freq -= 100000000;
               break;
            case '*':
               cur_freq += 100000000;
               break;

            //
            // Space: Reinitialize synthesizer
            //

            case ' ':

               delete synth;
               synth = NULL;
               break;

            //
            // A/Z: Frequency limits
            //

            case 'a':
            case 'A':
               cur_freq = FOUT_MIN;
               break;

            case 'z':
            case 'Z':
               cur_freq = FOUT_MAX;
               break;

            //
            // u/U: SA updates on/off
            // p/P: Take screenshot
            //

#if ANALYZER_CONTROL
            case 'u':
               SA_update = 0;
               printf("Tek 49xP GPIB updates off\n");
               break;

            case 'U':
               SA_update = 1;
               printf("Tek 49xP GPIB updates on\n");
               GPIB_update(cur_freq);
               break;

            case 'p':
            case 'P':

               GPIB_plot(cur_freq);
               break;
#endif

            //
            // Ctrl-A: Spur search mode
            //

            case '':

               spur_search_mode = !spur_search_mode;
               break;

            //
            // ESC: Exit
            //

            case 27:
               exit(0);
            }
         }
      }
}
