/* GRAF.C
 *
 *	Plots a graph on the display screen. Uses Borland graphics
 *	routines.
 *
 *	Copyright (c) 1995, American Radio Relay League
 *	This program can be used free of charge for noncommercial use.
 *	Contact the American Radio Relay League, Inc, 225 Main Street,
 *	Newington, CT 06111 (tel: 203-666-1541 x276) for information
 *	regarding commercial use.
 */

#include <stdio.h>
#include <stdlib.h>
#include <bios.h>
#include <ctype.h>
#include <graphics.h>
#include <math.h>
#include <mem.h>

#include "graf.h"

int usecolor = 1;

static int hofs, hlen, vofs, vlen;

extern char title[];	/* Graph title */

/*	r _ g r a p h
 *
 *	points	Array of values to plot
 *	n		Number of elements in "points"
 *	first	First element of "points" to plot
 *	minx	Minimum X-axis value (X value for first element of "points")
 *	maxx	Maximum X-axis value (X value for last element of "points")
 *	nx		Number of X-axis grid lines to plot
 *	miny	Minimum Y-axis value
 *	maxy	Maximum Y-axis value
 *	ny		Number of Y-axis grid lines to plot
 *	xlab	X-axis label
 *	ylab	Y-axis label
 */
int
r_graph(double *points, int n, int first, double minx, double maxx, int nx,
	double miny, double maxy, int ny, char *xlab, char *ylab)
{
	static int driver = DETECT, mode = 0;
	int h, v, last = 0;
	int i, j, prec;
	double xstep, ystep, val;
	char buf[120];
	static unsigned grid = 0x8888;

	if (driver == DETECT)
	{
		registerbgidriver(CGA_driver);
		registerbgidriver(EGAVGA_driver);
		registerbgidriver(Herc_driver);
		initgraph(&driver, &mode, getenv("GRAPHDIR"));
		if (driver < 0)
		{
			driver = DETECT;
			return -1;
		}
	} else
		setgraphmode(mode);

	/* Display title, if any */

	if (strlen(title) != 0) {
		hofs = (getmaxx() - textwidth(title)) / 2;
		outtextxy(hofs, 0, title);
	}
	hofs = textwidth("MMMMMMMMMMM");
	hlen = getmaxx() - hofs - textwidth("MMMMM");
	vofs = getmaxy() - textheight("M") * 8;
	vlen = vofs - textheight("M") * 7 / 2;
	xstep = (maxx - minx) / (nx - 1);
	ystep = (maxy - miny) / (ny - 1);
	rectangle(hofs, vofs, hofs + hlen, vofs - vlen);
	v = vofs + textheight("M") / 2;
	val = log10(maxx);
	if (val >= 3)
		prec = 0;
	else if (val < 0)
		prec = 3;
	else
		prec = 3 - val;
	
	/* Display X-axis grid lines and labels */
	
	for (i = 0; i < nx; i++)
	{
		sprintf(buf, "%.*lf", prec, minx + xstep * i);
		h = hofs + (i * hlen) / (nx - 1);
		if (h > last)
		{
			line(h, vofs, h, v);
			outtextxy(h - textwidth(buf) / 2, v + textheight("M") / 2, buf);
			last = h + textwidth(buf);
			if (i != 0 && i != nx-1)
			{
				setlinestyle(USERBIT_LINE, grid, NORM_WIDTH);
				line(h, vofs-1, h, vofs - vlen + 1);
				setlinestyle(SOLID_LINE, 0, NORM_WIDTH);
			}
		}
	}
	if (xlab != NULL) {
		h = hofs + hlen / 2 - textwidth(xlab) / 2;
		v = vofs + 3 * textheight("M");
		outtextxy(h, v, xlab);
	}
	h = hofs - textwidth("M") / 2;

	/* Display Y-axis grid lines and labels */

	for (i = 0; i < ny; i++)
	{
		val = miny + ystep * i;
		if (fabs(val) < 1e-5)
			val = 0;
		sprintf(buf, "%.*g", (val >= 1000) ? 0 : 3, val);
		v = vofs - (i * vlen) / (ny - 1);
		line(h, v, hofs, v);
		j = h - textwidth("M") / 2 - textwidth(buf);
		outtextxy(j, v - textheight("M") / 2, buf);
		if (i != 0 && i != ny-1)
		{
			setlinestyle(USERBIT_LINE, grid, NORM_WIDTH);
			line(hofs + 1, v, hofs + hlen - 1, v);
			setlinestyle(SOLID_LINE, 0, NORM_WIDTH);
		}
	}
	if (ylab != NULL) {
		h = textwidth("M");
		v = vofs - vlen / 2 - textwidth(ylab) / 2;
		settextstyle(DEFAULT_FONT, VERT_DIR, 1);
		outtextxy(h, v, ylab);
		settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
	}
	if (usecolor) {
		if (driver == CGA || driver == MCGA)
			setcolor(CGA_YELLOW);
		else if (driver == EGA || driver == EGA64 || driver == VGA)
			setcolor(YELLOW);
	}
	
	/* Display data */
	
	for (i = first; i < n; i++)
	{
		h = hofs + (int) (((long) hlen * i) / (n-1));
		val = (points[i] - miny) / (maxy - miny);
		v = vofs - (int) (val * vlen);
		if (v > vofs)
			v = vofs;
		if (i > first)
			line(j, last, h, v);
		j = h;
		last = v;
	}
	if (usecolor) {
		if (driver == CGA || driver == MCGA)
			setcolor(1);
		else if (driver == EGA || driver == EGA64 || driver == VGA)
			setcolor(LIGHTGRAY);
	}
	return 0;
}
