//	SwDDS60.cpp             note six changes for NJQRP DDS-60 at			*#*#*#*#*#*
//	written by Dr. Sam Green, W0PCE 
//	for the Fully Automated DDS Sweep Generator Measurement System 
/*
	This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
//	MAXIM MX7543 serial D to A Converter driver  
//	synthsizes  sawtooth sweep for oscilloscope horizontal deflection 
//  With hardware set up for bipolar operation, 
//  0x000 gives (2048/2048) x (-Vref)
//  0x800 gives 0 volts
//  0xfff gives (2047/2048) x (+Vref)
//  With Vref set by LM103 to 2.0 volts, range is from -2V to +1.999V 
//		or a resolution of about 1 mV 
//  Sweeps DDS output frequency and DAC output voltage 
//	to synthsize horizontal O'scope drive 
//  with discrete output to trigger oscilloscope sweep 
//
//	This program sweeps the DDS linearly from a lower frequency to a higher frequency while 
//	the D/A converter sweeps over a fixed range from the most negative voltage 
//	to the most positive voltage with the same number of steps.   
//	Set frequency start, end, and step number parameters in the command line.  
//	Default are 1 MHz to 10 MHz and 4095 steps.  
//  SWDDS.EXE [START] [END] [NUMBEROFSTEPS] 
//	with START and END in integer Hertz between  
//	and an integer NUMBEROFSTEPS
//	Numeric keypad allows crude increase and decrease of start frequency, end frequency, 
//	and  number of steps 

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <time.h>
#include <windows.h>

void delay (int time) ;
void delaym (long msecs) ;
void write40bit2d0 (int freqword, int phaseword) ;	// DDS
void gotoxy (int x, int y) ;
void changeparameters(int cc);
void write12databits2c0 (int dacword) ;				// DAC
void adjust (void) ;

int RefClock = 180000000;	// 180000000 for 180MHz clock on DDS-60			*#*#*#*#*#* 

const double mult = 0x100000000/RefClock;  

int base	= 0x378,		// parallel port register addresses
	status	= base+1,
	control	= base+2,
	freqstartword  = 1000000,	// default  1 MHz start
	phaseword = 0x01,		// phaseword = 0 for DDS-30 and 1 for DDS-60	*#*#*#*#*#* 
	freqendword  = 10000000, 	// default 10 MHz end
	numberofsteps = 0x1000,	// default 4096 steps 
	freqword = freqstartword,
	dacstart = 0x000,	// default -Vref start
	dacend   = 0xfff, 	// default Vref*2047/2048 end
	dacword = dacstart,
	dacstepcount = (dacend-dacstart)/numberofsteps  ;		// default dac increment 
	
float freqstepcount = (freqendword-freqstartword)/numberofsteps;	// default freq increment 

void main(int argc, char *argv[]) {
	char c; int cc; 
	system("CLS");
	freqword = freqstartword;  
	dacword = dacstart;  

	if (argc>1) {freqstartword = atoi(argv[1]);}
	if (argc>2)   {freqendword = atoi(argv[2]);} 
	if (freqendword < freqstartword) {puts("Enter end frequency higher than start frequency!");_exit(1);}
	if (argc>3) {numberofsteps = atoi(argv[3]); 
	} 

	adjust () ;
	gotoxy(3,1);printf("Usage: %s [start] [end] [#steps]\n", argv[0]); 
	while(1){								// swept frequency
		_outp(control, 8);_outp(control, 0);// pulse LPT pin 17 high & low to trigger o'scope  
		for(freqword = freqstartword; freqword < freqendword; freqword += freqstepcount) {
			write40bit2d0 (freqword, phaseword) ;

			dacword += dacstepcount; 
			write12databits2c0 (dacword) ;

			if(kbhit()) {(c = getch()); cc=c;changeparameters(cc);break;}
			gotoxy(1,3);printf("DDS-60"); // change 30 to 60 for DDS-60		*#*#*#*#*#*
			gotoxy(3,13);printf("\tInstantaneous Frequency = %8.1f\n", float(freqword));
			gotoxy(3,18);printf("\tDAC step count = 0x%x\n", dacstepcount);
			gotoxy(3,20);printf("\tInstantaneous Dacword = 0x%x\n", dacword);
//			delaym(1);  // milliseconds
		}	dacword = dacstart ; // end and restart DAC sweep too 
}	}
	
void delay (int time){ // short delay in fractions of a microsecond
	int i;
	for (i=0; i<time; i++) ;
}
	
void delaym (long msecs) {  // long delay in milliseconds
	clock_t now = clock();  // get current time
	clock_t then = now + CLOCKS_PER_SEC * (long)msecs/1000 ;
	while (now<then) now=clock() ;
}

void write40bit2d0 (int freqword, int phaseword) { // with clock to d1 & freq update strobe to d2
	int freqvar, i, writebit;

	freqvar = freqword*mult ;
	for(i=1; i<33; i++) {           
		writebit = freqvar & 1 ;	
		_outp(base, writebit);		// write bit
		_outp(base, writebit | 2);	// writeclock high
		_outp(base, writebit);	// writeclock low 
		freqvar >>= 1 ;
	}
	freqvar = phaseword ;  // all zeroes 
	for(i=1; i<9; i++) {           
		writebit = freqvar & 1 ;	
		_outp(base, writebit);		// write bit
		_outp(base, writebit | 2);	// writeclock high
		_outp(base, writebit);	// writeclock low 
		freqvar >>= 1 ;
	}
	_outp(base, 4);	// frequency update stobe pulse high;  
	_outp(base, 0);	// frequency update stobe pulse low;  
}

void write12databits2c0 (int dacword) { // with clock to c1 & transfer update strobe to c2 
	int dacvar, i, writebit, mask=0x1000;

	dacvar = ~dacword ;					// bitwise invert to write data to inverted C0 
	for(i=1; i<13; i++) {           
		mask >>= 1 ; writebit = dacvar & mask ; if (writebit!=0) writebit = 1; // reverses dacvar 
		_outp(base+2, 4 + writebit);	// write bit & keep both load (4) and clear (8) inactive
		_outp(base+2, 6 + writebit);	// writeclock low through inverted C1 & keep both load and clear inactive
		_outp(base+2, 4 + writebit);	// writeclock high through inverted C1 & keep both load and clear inactive
		}
	_outp(base+2, 0);					// update stobe pulse low noninverted C2
	_outp(base+2, 4);					// update stobe pulse high  
}

void gotoxy (int x, int y){
	HANDLE hdl;
	COORD coords;
	hdl = GetStdHandle(STD_OUTPUT_HANDLE);
	coords.X=x-1;
	coords.Y=y-1;
	SetConsoleCursorPosition(hdl,coords);
}

void changeparameters(int cc){
	gotoxy(12,15);
	switch (cc){
		case 0x37:  if (freqendword > 2.2*freqstartword) freqstartword *= 2;	// 7
					else freqstartword *= 1.1;
					if (freqendword <=freqstartword) freqstartword = 0.9*freqendword; 
					adjust(); break;	

		case 0x31:  freqstartword /= 2;											// 1
					adjust(); break;	
								//		 change 15 to 30 for DDS-60				*#*#*#*#*#*
		case 0x39:  if (freqendword < 30000000) freqendword *= 2  ;				// 9 
					adjust(); break;	

		case 0x33:  if (freqendword > 2.2*freqstartword) freqendword /= 2;		// 3
					else freqendword *= .9;
					if (freqendword <=freqstartword) freqendword = 1.1*freqstartword; 
					adjust(); break;	

		case 0x38:  numberofsteps/=2 ; 											// 8
					adjust(); break; 

		case 0x32:  numberofsteps*=2 ; 											// 2
					adjust(); break;	

		default:	;		
}	}


void adjust (void) {
	if (freqendword > 60000000) {	// change 3 to 6 in 3 places for DDS-60		*#*#*#*#*#*
		freqendword = 60000000;	// Comment out this line to remove upper limit	*#*#*#*#*#*
		gotoxy(50,5);puts ("60 MHz limit!");				//					*#*#*#*#*#*
		} 
	if (freqstartword < 1) freqstartword = 1;		
	if (numberofsteps > 4096) {numberofsteps = 4096; gotoxy(44,7); puts ("\t4096 step limit!");} // DAC steps  
	freqstepcount = (freqendword-freqstartword)/float(numberofsteps); 
	dacstepcount  = (dacend-dacstart)/numberofsteps; 
	if (dacstepcount <1) dacstepcount =1; 
	gotoxy(3,3);printf("\tStart Frequency\t\t= %8.0d Hz ", (freqstartword)); 
	gotoxy(3,5);printf("\tEnd Frequency\t\t= %8.0d Hz   ", (freqendword)); 
	gotoxy(3,7);printf("\tNumber of Steps\t\t= %8.0d    ",numberofsteps);
	gotoxy(3,9);printf("\tFrequency Increment\t= %8.1f Hz    \n", freqstepcount); 
	gotoxy(3,17);printf("\tSweep DAC from\t0x%.3x to 0x%x\n", dacstart,dacend);	
}