/****************************************************************************
 Title	:   HD44780U LCD library
 Author:    Peter Fleury <pfleury@gmx.ch>  http://jump.to/fleury
 File:	    $Id: lcd.c,v 1.13 2003/07/13 07:33:10 peter Exp $
 Software:  AVR-GCC 3.3 
 Target:    any AVR device, memory mapped mode only for AT90S4414/8515/Mega

 DESCRIPTION
       Basic routines for interfacing a HD44780U-based text lcd display

       Originally based on Volker Oth's lcd library,
       changed lcd_init(), added additional constants for lcd_command(),
       added 4-bit I/O mode, improved and optimized code.

       Library can be operated in memory mapped mode (LCD_IO_MODE=0) or in 
       4-bit IO port mode (LCD_IO_MODE=1). 8-bit IO port mode not supported.
       
       Memory mapped mode compatible with Kanda STK200, but supports also
       generation of R/W signal through A8 address line.
       
 Modified July 2004 by J.A. Koehler to be usable with two LCD displays, sharing all the 7
 lower lines and using pin x6 for Enable on display number 0 and x7 for display number 1.

 USAGE
       See the C include lcd.h file for a description of each function
       
*****************************************************************************/
#include <inttypes.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include "lcd.h"



/* 
** constants/macros 
*/
#define PIN(x) (*(&x - 2))  /* address of data direction register of port x */
#define DDR(x) (*(&x - 1))  /* address of input register of port x          */
#define lcd_space ' '
#define lcd_block 255


#if LCD_IO_MODE
#define lcd_e0_high()    LCD_E_PORT  |=  _BV(LCD_E0_PIN);
#define lcd_e0_low()     LCD_E_PORT  &= ~_BV(LCD_E0_PIN);
#define lcd_e0_toggle()  toggle_e0()
#define lcd_e1_high()    LCD_E_PORT  |=  _BV(LCD_E1_PIN);
#define lcd_e1_low()     LCD_E_PORT  &= ~_BV(LCD_E1_PIN);
#define lcd_e1_toggle()  toggle_e1()
#define lcd_rw_high()   LCD_RW_PORT |=  _BV(LCD_RW_PIN)
#define lcd_rw_low()    LCD_RW_PORT &= ~_BV(LCD_RW_PIN)
#define lcd_rs_high()   LCD_RS_PORT |=  _BV(LCD_RS_PIN)
#define lcd_rs_low()    LCD_RS_PORT &= ~_BV(LCD_RS_PIN)
#endif

#if LCD_IO_MODE
#if LCD_LINES==1
#define LCD_FUNCTION_DEFAULT    LCD_FUNCTION_4BIT_1LINE 
#else
#define LCD_FUNCTION_DEFAULT    LCD_FUNCTION_4BIT_2LINES 
#endif
#else
#if LCD_LINES==1
#define LCD_FUNCTION_DEFAULT    LCD_FUNCTION_8BIT_1LINE
#else
#define LCD_FUNCTION_DEFAULT    LCD_FUNCTION_8BIT_2LINES
#endif
#endif


/* 
** function prototypes 
*/
static void delay(uint16_t us);
static void lcd_e_delay(void);

#if LCD_IO_MODE
static void toggle_e0(void);
static void toggle_e1(void);
#endif

/*
** local functions
*/

/************************************************************************* 
delay for a minimum of <us> microseconds
*************************************************************************/
static void delay(uint16_t us) 
{
    while ( us-- ) lcd_e_delay();
}

/************************************************************************* 
delay for approximately 1 microsecond with a 14 MHz clock
*************************************************************************/
static void lcd_e_delay(void)
{
asm volatile
  (
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  "nop" "\n"
  );
}

#if LCD_IO_MODE
/* toggle Enable Pin to initiate write */
static void toggle_e0(void)
{
    lcd_e0_high();
    delay(5);
    lcd_e0_low();
}

/* toggle Enable Pin to initiate write */
static void toggle_e1(void)
{
    lcd_e1_high();
    delay(5);
    lcd_e1_low();
}
#endif


/*************************************************************************
Low-level function to write byte to LCD controller
Input:    data   byte to write to LCD
          rs     1: write data    
                 0: write instruction
Returns:  none
*************************************************************************/
#if LCD_IO_MODE
static void inline lcd_write(uint8_t data,uint8_t rs, uint8_t LCD_number) 
{
    /* configure data pins as output */
    DDR(LCD_DATA_PORT) = 0xFF;

    if (rs) {   /* write data        (RS=1, RW=0) */
    
       /* output high nibble first */
       LCD_DATA_PORT = ((data>>4)&0x0F)|(1<<LCD_RS_PIN);
       if (LCD_number == 0) lcd_e0_toggle(); else lcd_e1_toggle();

       /* output low nibble */
       LCD_DATA_PORT = (data&0x0F)|(1<<LCD_RS_PIN);
       if (LCD_number == 0) lcd_e0_toggle(); else lcd_e1_toggle();
       
    } else {    /* write instruction (RS=0, RW=0) */
           
       /* output high nibble first */
       LCD_DATA_PORT = (data>>4)&0x0F;
       if (LCD_number == 0) lcd_e0_toggle(); else lcd_e1_toggle();

       /* output low nibble */
       LCD_DATA_PORT = data&0x0F;
       if (LCD_number == 0) lcd_e0_toggle(); else lcd_e1_toggle();
    }

    /* all data pins high (inactive) */
    LCD_DATA_PORT = 0x0F;
}
#else
#define lcd_write(d,rs) if (rs) *(volatile uint8_t*)(LCD_IO_DATA) = d; else *(volatile uint8_t*)(LCD_IO_FUNCTION) = d;
/* rs==0 -> write instruction to LCD_IO_FUNCTION */
/* rs==1 -> write data to LCD_IO_DATA */
#endif


/*************************************************************************
Low-level function to read byte from LCD controller
Input:    rs     1: read data    
                 0: read busy flag / address counter
Returns:  byte read from LCD controller
*************************************************************************/
#if LCD_IO_MODE
static uint8_t lcd_read(uint8_t rs, uint8_t LCD_number) 
{
    register uint8_t dataH, dataL;

    if (rs)
        lcd_rs_high();                       /* RS=1: read data      */
    else
        lcd_rs_low();                        /* RS=0: read busy flag */
    lcd_rw_high();                           /* RW=1  read mode      */
    
    /* configure data pins as input */
    DDR(LCD_DATA_PORT) = 0xF0;
    
    if (LCD_number == 0)
      {
      lcd_e0_high();
      lcd_e_delay();
      dataH = PIN(LCD_DATA_PORT);              /* read high nibble first */
      lcd_e0_low()
      }
    else
      {
      lcd_e1_high();
      lcd_e_delay();
      dataH = PIN(LCD_DATA_PORT);              /* read high nibble first */
      lcd_e1_low()
      }
      
    lcd_e_delay();                           /* Enable 500ns low       */
    
    if (LCD_number == 0)
      {
      lcd_e0_high();
      lcd_e_delay();
      dataL = PIN(LCD_DATA_PORT);              /* read low nibble next */
      lcd_e0_low()
      }
    else
      {
      lcd_e1_high();
      lcd_e_delay();
      dataL = PIN(LCD_DATA_PORT);              /* read low nibble next */
      lcd_e1_low()
      }
    
    return ( (dataH<<4) | (dataL&0x0F) );
}
#else
#define lcd_read(rs) (rs) ? *(volatile uint8_t*)(LCD_IO_DATA+LCD_IO_READ) : *(volatile uint8_t*)(LCD_IO_FUNCTION+LCD_IO_READ)
/* rs==0 -> read instruction from LCD_IO_FUNCTION */
/* rs==1 -> read data from LCD_IO_DATA */
#endif


/*************************************************************************
loops while lcd is busy, returns address counter
*************************************************************************/
static uint8_t lcd_waitbusy(uint8_t LCD_number)

{
    register uint8_t c;
    
    /* wait until busy flag is cleared */
    while ( (c=lcd_read(0, LCD_number)) & (1<<LCD_BUSY)) {}
    
    /* the address counter is updated 4us after the busy flag is cleared */
    delay(2);

    /* now read the address counter */
    return (lcd_read(0, LCD_number));  // return address counter
    
}/* lcd_waitbusy */


/*************************************************************************
Move cursor to the start of next line or to the first line if the cursor 
is already on the last line.
*************************************************************************/
static inline void lcd_newline(uint8_t pos, uint8_t LCD_number)
{
    register uint8_t addressCounter;


#if LCD_LINES==1
    addressCounter = 0;
#endif
#if LCD_LINES==2
    if ( pos < (LCD_START_LINE2) )
        addressCounter = LCD_START_LINE2;
    else
        addressCounter = LCD_START_LINE1;
#endif
#if LCD_LINES==4
    if ( pos < LCD_START_LINE3 )
        addressCounter = LCD_START_LINE2;
    else if ( (pos >= LCD_START_LINE2) && (pos < LCD_START_LINE4) )
        addressCounter = LCD_START_LINE3;
    else if ( (pos >= LCD_START_LINE3) && (pos < LCD_START_LINE2) )
        addressCounter = LCD_START_LINE4;
    else 
        addressCounter = LCD_START_LINE1;
#endif
    lcd_command((1<<LCD_DDRAM)+addressCounter, LCD_number);

}/* lcd_newline */


/*
** PUBLIC FUNCTIONS 
*/

/*************************************************************************
Send LCD controller instruction command
Input:   instruction to send to LCD controller, see HD44780 data sheet
Returns: none
*************************************************************************/
void lcd_command(uint8_t cmd, uint8_t LCD_number)
{
    lcd_waitbusy(LCD_number);
    lcd_write(cmd, 0, LCD_number);
}


/*************************************************************************
Set cursor to specified position
Input:    x  horizontal position  (0: left most position)
          y  vertical position    (0: first line)
Returns:  none
*************************************************************************/
void lcd_gotoxy(uint8_t x, uint8_t y, uint8_t LCD_number)
{
#if LCD_LINES==1
    lcd_command((1<<LCD_DDRAM)+LCD_START_LINE1+x, LCD_number);
#endif
#if LCD_LINES==2
    if ( y==0 ) 
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE1+x, LCD_number);
    else
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE2+x, LCD_number);
#endif
#if LCD_LINES==4
    if ( y==0 )
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE1+x, LCD_number);
    else if ( y==1)
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE2+x, LCD_number);
    else if ( y==2)
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE3+x, LCD_number);
    else /* y==3 */
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE4+x, LCD_number);
#endif

}/* lcd_gotoxy */


/*************************************************************************
*************************************************************************/
int lcd_getxy(uint8_t LCD_number)
{
    return lcd_waitbusy(LCD_number);
}


/*************************************************************************
Clear display and set cursor to home position
*************************************************************************/
void lcd_clrscr(uint8_t LCD_number)
{
    lcd_command(1<<LCD_CLR, LCD_number);
}


/*************************************************************************
Set cursor to home position
*************************************************************************/
void lcd_home(uint8_t LCD_number)
{
    lcd_command(1<<LCD_HOME, LCD_number);
}


/*************************************************************************
Display character at current cursor position 
Input:    character to be displayed                                       
Returns:  none
*************************************************************************/
void lcd_putc(char c, uint8_t LCD_number)
{
    uint8_t pos;


    pos = lcd_waitbusy(LCD_number);   // read busy-flag and address counter
    if (c=='\n')
    {
        lcd_newline(pos, LCD_number);
    }
    else
    {
#if LCD_WRAP_LINES==1
#if LCD_LINES==1
        if ( pos == LCD_START_LINE1+LCD_DISP_LENGTH )
            lcd_write((1<<LCD_DDRAM)+LCD_START_LINE1,0, LCD_number);
#elif LCD_LINES==2
        if ( pos == LCD_START_LINE1+LCD_DISP_LENGTH )
            lcd_write((1<<LCD_DDRAM)+LCD_START_LINE2,0, LCD_number);    
        else if ( pos == LCD_START_LINE2+LCD_DISP_LENGTH )
            lcd_write((1<<LCD_DDRAM)+LCD_START_LINE1,0, LCD_number);
#elif LCD_LINES==4
        if ( pos == LCD_START_LINE1+LCD_DISP_LENGTH )
            lcd_write((1<<LCD_DDRAM)+LCD_START_LINE2,0, LCD_number);    
        else if ( pos == LCD_START_LINE2+LCD_DISP_LENGTH )
            lcd_write((1<<LCD_DDRAM)+LCD_START_LINE3,0, LCD_number);
        else if ( pos == LCD_START_LINE3+LCD_DISP_LENGTH )
            lcd_write((1<<LCD_DDRAM)+LCD_START_LINE4,0, LCD_number);
        else if ( pos == LCD_START_LINE4+LCD_DISP_LENGTH )
            lcd_write((1<<LCD_DDRAM)+LCD_START_LINE1,0, LCD_number);
#endif
        lcd_waitbusy(LCD_number);
#endif
        lcd_write(c, 1, LCD_number);
    }

}/* lcd_putc */


/*************************************************************************
Display string without auto linefeed 
Input:    string to be displayed
Returns:  none
*************************************************************************/
void lcd_puts(const char *s, uint8_t LCD_number)
/* print string on lcd (no auto linefeed) */
{
    register char c;

    while ( (c = *s++) ) {
        lcd_putc(c, LCD_number);
    }

}/* lcd_puts */


/*************************************************************************
Display string from program memory without auto linefeed 
Input:     string from program memory be be displayed                                        
Returns:   none
*************************************************************************/
void lcd_puts_p(const char *progmem_s, uint8_t LCD_number)
/* print string from program memory on lcd (no auto linefeed) */
{
    register char c;

    while ( (c = pgm_read_byte(progmem_s++)) ) {
        lcd_putc(c, LCD_number);
    }

}/* lcd_puts_p */

void lcd_barograph(float v, uint8_t line, uint8_t LCD_number)
 /*
 
 	This routine draws a bar graph on line numbered 'line' and on LCD number, LCD_number.
The bar extends from 0 to the number of characters in the line
in going from 0 to 1.0.  The value, v, is normalized to be between 0 and 1.0

Modified Feb, 2005 to give a bar with maximum length of 100
 */
  
  {
  uint8_t i, j;
  
  lcd_gotoxy(0, line, LCD_number);
  
/*
  if (v < 0.0) v = -v;
  

  if (v == 0.0) v = 0.1001;
  
  while (v > 1.0) v /= 10.0;
  while (v < 0.1) v *= 10.0;


//v is now between 0 and 1.0

  i = (int8_t) (v * LCD_DISP_LENGTH);
  
*/
  i = (uint8_t) ((v / 100.0) * LCD_DISP_LENGTH);
  j = LCD_DISP_LENGTH - i;
  
  if (i >= LCD_DISP_LENGTH)
    {
    i = LCD_DISP_LENGTH;
    j = 0;
    }
// Now write the line 
  while (i--) lcd_putc((char) lcd_block, LCD_number);
  while (j--) lcd_putc((char) lcd_space, LCD_number);
  }
/*************************************************************************
Initialize display and select type of cursor 
Input:    dispAttr LCD_DISP_OFF            display off
                   LCD_DISP_ON             display on, cursor off
                   LCD_DISP_ON_CURSOR      display on, cursor on
                   LCD_DISP_CURSOR_BLINK   display on, cursor on flashing
Returns:  none
*************************************************************************/
void lcd_init(uint8_t dispAttr, uint8_t LCD_number)
{
#if LCD_IO_MODE   
    /*
     *  Initialize LCD to 4 bit I/O mode
     */
         
    DDR(LCD_DATA_PORT) = 0xFF;          /* all port bits as output */    
    delay(16000);        /* wait 16ms or more after power-on       */
    
    /* initial write to lcd is 8bit */
    LCD_DATA_PORT= LCD_FUNCTION_8BIT_1LINE>>4;
    
    if (LCD_number == 0) lcd_e0_toggle(); else lcd_e1_toggle();
    delay(4992);         /* delay, busy flag can't be checked here */
    
    LCD_DATA_PORT = LCD_FUNCTION_8BIT_1LINE>>4;
    if (LCD_number == 0) lcd_e0_toggle(); else lcd_e1_toggle();
    delay(64);           /* delay, busy flag can't be checked here */
    
    LCD_DATA_PORT = LCD_FUNCTION_8BIT_1LINE>>4;
    if (LCD_number == 0) lcd_e0_toggle(); else lcd_e1_toggle();
    delay(64);           /* delay, busy flag can't be checked here */
    
    LCD_DATA_PORT = LCD_FUNCTION_4BIT_1LINE>>4; /* set IO mode to 4bit */
    if (LCD_number == 0) lcd_e0_toggle(); else lcd_e1_toggle();
    delay(64);           /* some displays need this additional delay */
    
    /* from now the LCD only accepts 4 bit I/O, we can use lcd_command() */    
#else
    /*
     * Initialize LCD to 8 bit memory mapped mode
     */
    
    /* enable external SRAM (memory mapped lcd) and one wait state */        
    MCUCR = _BV(SRE) | _BV(SRW);

    /* reset LCD */
    delay(16000);                           /* wait 16ms after power-on     */
    lcd_write(LCD_FUNCTION_8BIT_1LINE,0, uint8_t LCD_number);   /* function set: 8bit interface */                   
    delay(4992);                            /* wait 5ms                     */
    lcd_write(LCD_FUNCTION_8BIT_1LINE,0, uint8_t LCD_number);   /* function set: 8bit interface */                 
    delay(64);                              /* wait 64us                    */
    lcd_write(LCD_FUNCTION_8BIT_1LINE,0, uint8_t LCD_number);   /* function set: 8bit interface */                
    delay(64);                              /* wait 64us                    */
#endif
    lcd_command(LCD_FUNCTION_DEFAULT, LCD_number);      /* function set: display lines  */
    lcd_command(LCD_DISP_OFF,LCD_number);              /* display off                  */
    lcd_clrscr(LCD_number);                           /* display clear                */ 
    lcd_command(LCD_MODE_DEFAULT, LCD_number);          /* set entry mode               */
    lcd_command(dispAttr, LCD_number);                  /* display/cursor control       */

}/* lcd_init */
