// Version JT65V006
// 22-November-2013
// Modified for KW5GP Frog Sounds V3 QRP CW Transceiver Board
// 7-January-2015
// Built on the Ten-Tec Rebel A0.2 clean firmware with EVERYTHING not
// necessary for this one specific purpose removed.
//
// Uses modified QRP SkunkWerks Ten-Tec Rebel JT65 firmware and HFWST-Phoenix
// for the Ten-Tec Rebel by Joe Large, W6CQZ.
// Includes JT9 code (not implemented) for HFWST - removing it breaks HFWST
//
// Uses M0XPD DDS Libary for the AD9850
//
// Emulates 40m Ten-Tec JT65 Rebel with Nokia 5110 Display

// Get all the includes up top - seems to fix some strange
// compiler issues if I do it here.

#include <Streaming.h>  // Needed by CmdMessenger
#include <morse.h>  // Morse Library for CW ID
#include <CmdMessenger.h>  // Command Messenger to communicate with HFWST
#include <DDS.h>  // M0XPD DDS Library

// Nokia 5110 Pin Assignments
#define GLCD_SCK   30
#define GLCD_MOSI  29
#define GLCD_DC    28
#define GLCD_RST   26
#define GLCD_CS    27

// Nokia 5110 LCD Display Library
#include <LCD5110_Basic.h>

// Nokia 5110 LCD Library Pin Definitions
LCD5110 glcd(GLCD_SCK,GLCD_MOSI,GLCD_DC,GLCD_RST,GLCD_CS);
  
extern unsigned char SmallFont[];

#pragma message("Compiling with Nokia Support");

// various defines

#define TX_OUT 38 //  TX Key out pin
#define TX_LED 8  // TX Key LED

//======================================
// AD9850 Module....
// set pin numbers:
const int W_CLK = 13;
const int FQ_UD = 10; 
const int DATA = 11;
const int RESET = 9;

// Instantiate the DDS...
DDS dds(W_CLK, FQ_UD, DATA, RESET);

const int ROMVERSION        = 1005; // Defines this firmware revision level - not bothering with major.minor 0 to max_int_value "should" be enough space. :)

int TX_key;  // Variable used to control TX keying
int Battery_pin = A3; // 12v Input pin
float conversion = 0.0159255; // voltage conversion factor
float volt_drop = 01.04; // voltage drop across input diode and circuit loss

//------------------------------------------------------------
// Once tested as viable this will likely change to 9MHz = LO + RX RF for LO range of 2.0 MHz [RX 7.0] ... 1.7 [RX 7.3]
// Or if you'd rather 9-Desired RX = LO = 9-7.0=2 ... 9-7.3=1.7 for low side injection to get RX USB referenced at IF.
// Only works on paper so far.

const long meter_40             = 1924000;     // IF - Band frequency, LOW Side Inject JT65
// LOW side injection 40 meter 
// range 9 - 7 ... 9 - 7.3 -> 2.0 ... 1.7

const long Reference            = 49999750;     // for ad9834 this may be tweaked in software to fine tune the Radio

double ref_calc = pow(2,28)/Reference;  // Calculate conversion value from tuning word to frequency
double FR_0;   // Frequency Registers 0 to emulate AD9834 dual registers
double FR_1;    // Frequency Register 1 to emulate AD9834 dual registers
double dds_RX;  // RX frequency from HFWST

long frequency                  = 0;
long frequency_default          = 0;
unsigned long fcalc0;                               // Tuning word register 0 0 seems to be RX
long IF                         = 9.00e6;          //  I.F. Frequency  // keep in case used with an RX with IF

//------------------------------------------------------------
// Debug Stuff for HFWST
unsigned long loopCount         = 0;
unsigned long lastLoopCount     = 0;
unsigned long loopsPerSecond    = 0;
unsigned int  printCount        = 0;
unsigned long loopStartTime     = 0;
unsigned long loopElapsedTime   = 0;
float         loopSpeed         = 0;
unsigned long LastFreqWriteTime = 0;

//-------------------------------------------------------------------- 
// 10-10-2013 W6CQZ
// Adding array to hold transmit FSK values and handler for cmdMessenger serial control library
unsigned long fsk65Vals[128];  // 126 JT65 symbols + 2 spares :) See notes in loader code for +2 logic.
boolean jt65TXOn = false; // If true immediately start sending FSK set in fsk65Vals[0..125]
boolean jt65Valid = false; // Do NOT attempt JT mode TX unless this is true - remains false until a valid TX set is uploaded.
unsigned int jt65Sym = 0; // Index to where we are in the symbol TX chain

unsigned long fsk9Vals[88]; // ## JT9 fsk symbols - same logic as JT65 - uploading FSK vals in chunks of 4 so buffer needs to be multiple 4 (3 spares)
boolean jt9TXOn = false; // If true immediately start sending FSK set in fsk65Vals[0..125]
boolean jt9Valid = false; // Do NOT attempt JT mode TX unless this is true - remains false until a valid TX set is uploaded.
unsigned int jt9Sym = 0; // Index to where we are in the symbol TX chain

int rxOffset = 718; // Value to offset RX for correction DO NOT blindly trust this will be correct for your Rebel.
int txOffset = -50; // Value to offset TX for correction DO NOT blindly trust this will be correct for your Rebel.

boolean flipflop = true; // Used for ping-pong cycling FSK Values
long symoffset = 0;      // Begin at this Index on start TX

boolean txstat = false;  // Status of TX

CmdMessenger cmdMessenger = CmdMessenger(Serial);  // Serial handler

// Commands for rig control
enum
{
  kError,
  kAck,
  gVersion,
  gDDSVer,
  gDDSRef,
  sLockPanel,
  sUnlockPanel,
  gloopSpeed,
  gRXOffset,
  gTXOffset,
  sRXOffset,
  sTXOffset,
  gBand,
  gRXFreq,
  sRXFreq,
  gTXStatus,
  sTXOn,
  sTX9On,
  sTXOff,
  sDTXOn,
  sD9TXOn,
  sDoCWID,
  gClearTX,
  sTXFreq,
  gLoadTXBlock,
  gLoad9TXBlock,
  gFSKVals,
  gFSK9Vals,
  gGPSGrid,
  gGPSTime,
};
// Define the command callback routines
void attachCommandCallbacks()
{
  cmdMessenger.attach(OnUnknownCommand);              // Catch all in case of garbage/bad command - does nothing but ignore junk.                                              Command ID
  cmdMessenger.attach(gVersion, onGVersion);           // Get firmware version                                                                                           2
  cmdMessenger.attach(gDDSVer, onGDDSVer);             // Get DDS type                                                                                                   3
  cmdMessenger.attach(gDDSRef, onGDDSRef);             // Get DDS reference QRG                                                                                          4
  cmdMessenger.attach(sLockPanel, onSLockPanel);       // Lock out controls (going away - default is locked and stay locked)                                             5
  cmdMessenger.attach(sUnlockPanel, onSUnlockPanel);   // Unlock panel controls (use with caution!)                                                                      6
  cmdMessenger.attach(gloopSpeed, onLoopSpeed);        // Get main loop execution speed as string                                                                        7
  cmdMessenger.attach(gRXOffset, onGRXOffset);         // Get RX offset value                                                                                            8
  cmdMessenger.attach(gTXOffset, onGTXOffset);         // Get TX offset value                                                                                            9
  cmdMessenger.attach(sRXOffset, onSRXOffset);         // Set RX offset for correcting CW RX offset built into 2nd LO/mixer                                             10
  cmdMessenger.attach(sTXOffset, onSTXOffset);         // Set TX offset (usually 0 but if you want to calibrate the Rebel this value will do it, not the RX offset!     11
  cmdMessenger.attach(gBand, onGBand);                 // Get Band                                                                                                      12
  cmdMessenger.attach(gRXFreq, onGRXFreq);             // Get RX QRG                                                                                                    13
  cmdMessenger.attach(sRXFreq, onSRXFreq);             // Set RX QRG with DDS tuning word                                                                               14
  cmdMessenger.attach(gTXStatus, onGTXStatus);         // Get TX status, on or off - JT65 or JT9                                                                        15
  cmdMessenger.attach(sTXOn, onSTXOn);                 // Start TX - JT65                                                                                               16
  cmdMessenger.attach(sTX9On, onS9TXOn);               // Start TX - JT9                                                                                                17
  cmdMessenger.attach(sTXOff, onSTXOff);               // Stop TX - JT65 or JT9                                                                                         18
  cmdMessenger.attach(sDTXOn, onSDTXOn);               // Begin delayed TX at offset given - JT65                                                                       19
  cmdMessenger.attach(sD9TXOn, onSD9TXOn);             // Begin delayed TX at offset given - JT9                                                                        20
  cmdMessenger.attach(sDoCWID, onDoCWID);              // Send CW ID with string provided after current JT65 or JT9 TX is completed                                     21
  cmdMessenger.attach(gClearTX, onGClearTX);           // Clear FSK tuning word array - Clears JT9 and JT65 FSK Array                                                   22
  cmdMessenger.attach(sTXFreq, onSTXFreq);             // Request to setup TX array - JT65                                                                              23
  cmdMessenger.attach(gLoadTXBlock, onGLoadTXBlock);   // FSK tuning word loader setup - JT65 format                                                                    24
  cmdMessenger.attach(gLoad9TXBlock, onG9LoadTXBlock); // FSK tuning word loader - JT9 format                                                                           25
  cmdMessenger.attach(gFSKVals, onGFSKVals);           // Return current loaded FSK array - JT65                                                                        26
  cmdMessenger.attach(gFSK9Vals, onG9FSKVals);         // Return current loaded FSK array - JT9                                                                         27
  cmdMessenger.attach(gGPSGrid, onGGPSGrid);           // Get Grid from GPS                                                                                             28
  cmdMessenger.attach(gGPSTime, onGGPSTime);           // Get Time from GPS                                                                                             29
}
// --- End of cmdMessenger definition/setup ---

void setup() 
{
  for(jt65Sym=0; jt65Sym<128; jt65Sym++) {fsk65Vals[jt65Sym]=0;}  // 126 JT65 symbols + 2 spares :) See notes in loader code for +2 logic.
  for(jt9Sym=0; jt9Sym<88; jt9Sym++) {fsk9Vals[jt9Sym]=0;} // 85 JT9 symbols + 3 spares.  Same logic on +3 as above.
  jt65TXOn = false; // If true immediately start sending FSK set in fskVals[0..125]
  jt65Valid = false; // Do NOT attempt JT mode TX unless this is true - remains false until a valid TX set is uploaded.
  jt9TXOn = false; // If true immediately start sending FSK set in fskVals[0..125]
  jt9Valid = false; // Do NOT attempt JT mode TX unless this is true - remains false until a valid TX set is uploaded.
  jt65Sym = 0; // Index to where we are in the symbol TX chain
  jt9Sym = 0; // Index to where we are in the symbol TX chain
  rxOffset = 0; // 700 seems the nominal so far to correct for built in CW beat note offset in 2nd LO
  txOffset = 0; // Value to offset TX for correction. Can be changed via command.  -50 seems pretty common so far.
  flipflop = true; // Testing something
  
  pinMode (TX_OUT, OUTPUT);  // Set the TX Key pin to output
  digitalWrite (TX_OUT, LOW);  // Turn off TX Keying
  pinMode (TX_LED, OUTPUT);  // Set the TX Key pin to output
  digitalWrite (TX_LED, LOW);  // Turn off TX LED


  // display Startup Display
  glcd.InitLCD();  // Initialize the Nokia Display
  glcd.setContrast(65);    // Contrast setting of 65 looks good at 3.3v
  glcd.setFont(SmallFont);
  glcd.clrScr();
  glcd.print("KW5GP",CENTER, 0);
  glcd.print("Frog Sounds",CENTER,16);
  glcd.print("40m QRP Xcvr", CENTER,24);
  glcd.print("JT65 Version",CENTER,40);
  delay(5000);
  
  // Clear the LCD while waiting for HFWST to connect
  glcd.clrScr();
  glcd.print("Waiting for PC",CENTER,24);
    
  FR_0 = 7076000 + rxOffset;  // Go ahead and set to default 40M JT65 QRG
  FR_1 = 7076000 + txOffset;  // Go ahead and set to default 40M JT65 QRG
  dds_RX = FR_0;  // Load the RX freq into dds_RX so we can return to RX freq
  
  Default_Settings();  // Set the default settings

  dds.init();  // Setup the AD9850
  dds.trim(125000100); // AD9850 actual osc reference freq   
  
  digitalWrite(TX_OUT, LOW); // turn off TX Key
  digitalWrite(TX_LED, LOW); // Turn off TX LED
 
  attachCoreTimerService(TimerOverFlow);//See function at the bottom of the file.

  Serial.begin(115200);  // Fire up serial port (For HFWST this ***must*** be 9600 or 115200 baud)
  cmdMessenger.printLfCr(); // Making sure cmdMessenger terminates responses with CR/LF
  attachCommandCallbacks(); // Enables callbacks for cmdMessenger
  cmdMessenger.sendCmd(kAck,"W6CQZ_FW_1000");  // Sends a 1 time signon message at firmware startup
}

//    end of setup

//===================================================================
void Default_Settings()
{
  digitalWrite (TX_OUT, LOW); // Turn off TX
  digitalWrite (TX_LED, LOW); // Turn off TX_LED
  dds.setFrequency(FR_0);  // Set DDS to RX frequency in FR_0
}

//======================= Main Part =================================
void loop()     // 
{
  // I've setup cmdMessenger such that it should ignore anything that
  // makes no sense to it - but - if dealing with mixed data on serial
  // probably be best to process all that before calling feedinSerialData()
  // Wish we had two serial ports as we *should* :(  Will just have to
  // orchestrate points where serial belongs to GPS or command.
  
  // Process any serial data for commands
  cmdMessenger.feedinSerialData();
  
  if(jt65TXStatus())  // If there is a JT65 message to send
  {
    if(jt65FrameStatus())  // If we have a valid frame to send
    {
      int i=0;
      int j=0;
      int k=0;
      unsigned long rx = getRX();  // Save the RX frequency so we can return to it
      // Mark TX status = on
      txstat = true;
      
      flipflop = false; // Sets software "flipflop" to false where it needs to be for first symbol TX
      // Get correct value in place for first symbol to TX
      // program_ab loads register 0 and 1 in one pass.  Pass a 0 value to either if you only want to set 0 or 1.
      // program_ab takes TUNING WORDS NOT frequency values.  It then splits out the 2 14 bit tuning nibbles and
      // sends to DDS.
      // Adding a symbol offset to allow late TX start - this is RESET TO ZERO after TX cycle
      // The AD9834 DDS has two frequency registers. Code has been modified for use with the AD9850 and the dual
      // frequency registers are emulated by FR_0 and FR_1 and the tuning word is converted to real frequency
      // for the DDS Library
      
      program_ab(fsk65Vals[0+symoffset],fsk65Vals[1+symoffset]);
      i = 0+symoffset;
      j = j+symoffset;
      k = 0;
      for(i; i<126; i++)
      {
        if(k==0)
        {
          // Double++++++++ make sure FR zero is active and let free the blistering 5 watts upon the world
        
          // set DDS freq to FR0
          dds.setFrequency(FR_0);
          
          // Key external PTT before setting TX on
          digitalWrite(TX_OUT, HIGH); // Frightening little bit (for now cause this is the great unknown)
          digitalWrite(TX_LED, HIGH); // Turn on TX LED
          k++;
        }
        // OK - time to get in the trenches and make this happen.  Here's the process flow.
        // At start of TX preserve current RX value - load in first symbol value (fskVals[0]) to register 1
        // start the actual TX and *immediately* load register 0 with next value.  After delay switch register
        // to 0 and *immediately* load register 1 with next value.  Rinse and repeat until all 126 out the door
        // or an abort command is received from host (or *eventually* panel button press). When TX is done,
        // one way or another, restore RX LO value to register 0.  Done.
        //
        // One more time to make sure I keep my logic in line
        // At entry I have first 2 tones in register 0 and register 1.  Need to be sure register 0 Z E R O is
        // active as it containst the first tone. The software flipflop toggles after each delay.  If it is
        // false we TX from 0 load to 1.  If it is true we TX from 1 load to 0.
        // WARNING WARNING WARNING DO NOT NOT NOT clobber flipflop or bad bad things will happen during a TX
        // cycle.
        if(flipflop)
        {
          // Flipflop is true so we TX from Register 1 load to Register 0.
        
          // Set TX register to 1
          // Set the DDS frequency to FR_1
          dds.setFrequency(FR_1);  // Set Frequency to FR_1
          
          // Load in next value for register 0 leaving register 1 alone
          program_ab(fsk65Vals[j],0);
          j++;
        }
        else
        {
          // Flipflop is false so we TX from register 0 load to register 1.
         
          // Set TX register to 0
          // Set the DDS frequency to FR_0           
          dds.setFrequency(FR_0);  // Set Frequency to FR_0
          
          // Load in next value for register 1 leaving register 0 alone          
          program_ab(0,fsk65Vals[j]);
          j++;
        }
        // Maybe not best idea... frame time is actually 371.5193 mS.  372 is running us 63mS long by end of frame.
        // using this I do 63 at 372, 63 at 371 for a frame time of 46.809 seconds.  If I did the actual "real" JT65
        // symbol time length frame would be 46.811 versus 46.872 doing it all at 372 or 46.746 at 371.
        //
        // Error value at 372 = 46.872/46.811428571428571428571428571429 = 1.0012939453125 11010.8 Samples/Second equiv     
        // Error value at 371 = 46.746/46.811428571428571428571428571429 = 0.9986022949219 11040.4
        // Error value at mix = 46.809/46.811428571428571428571428571429 = 0.9999481201172 11025.6
        //
        // Compared to what's seen with typical sound cards doing this AFSK.... any of those would be beyond fine.  :)
        //
        //  If it breaks look here.  :)
        
        if(flipflop) { delay(372); } else { delay(371); }
        // CRTICIAL that this is kept right :)
        if(flipflop)
        {
          flipflop = false;
        }
        else
        {
          flipflop = true;
        }
        // Quick call to command parser so we could catch a TX abort.  This MUST MUST MUST be left
        // as is.  There's no way to stop TX without it short of yanking power.  
        
        cmdMessenger.feedinSerialData();
        if(!jt65TXStatus())
        {
          // Got TX abort
          // DROP TX NOW
          symoffset=0;
          digitalWrite(TX_OUT, LOW);  // Turn off TX Keying
          digitalWrite(TX_LED, LOW); // Turn off TX LED

          // Restore RX QRG
          program_ab(rx, 0);         
          dds.setFrequency(FR_0);    // Set DDS frequency to FR_0 (Original RX frequency)
          // Mark TX status = off
          txstat = false;
          break;
        }
      }
      // Clean up and restore RX
      // Drop TX NOW
      symoffset=0;
      digitalWrite(TX_OUT, LOW);  // Turn off TX Keying
      digitalWrite(TX_LED, LOW); // Turn off TX LED
      program_ab(rx, 0);

      dds.setFrequency(FR_0);   // Set Frequency to FR_0 (Original Rx frequency)
      // Mark TX status = off
      txstat = false;
    }
    else
    {
      // Frame did not validate
      digitalWrite(TX_OUT, LOW); // Make sure TX keying is off just to be safe :)
      digitalWrite(TX_LED, LOW); // Turn off TX LED      

      dds.setFrequency(FR_0);   // Set DDS frequency to FR_0
      stx65(false); // Set jtTXStatus false since the FSK values don't make sense.
      // Mark TX status = off
      txstat = false;
    }
    stx65(false);
  }
  else
  {
    digitalWrite(TX_OUT, LOW); // Make sure TX keying is off just to be safe :)
    digitalWrite(TX_LED, LOW); // Turn off TX LED    

    dds.setFrequency(dds_RX);   // Set Frequency to Original RX
    // Mark TX status = off
    txstat = false;
  }

  // Keep track of loop speed
  loopCount++;
  loopElapsedTime    = millis() - loopStartTime;
  // has 1000 milliseconds elasped?
  if( 1000 <= loopElapsedTime )
  {
    serialDump();    // comment this out to remove the one second tick
  }

}    //  END LOOP

//===================================================================

// ==================================================================
// ------------- Standard Rebel Functions ---------------------------

//--------------------Default Frequency-----------------------------------------
void Default_frequency()
{
  frequency = frequency_default;  // Set the default DDS frequency

  dds.setFrequency(frequency);  // Set the default DDS frequency

}   //  end   Default_frequency

//------------------ Debug data output ------------------------------
void    serialDump()  // HFWST Debug code
{
  loopStartTime   = millis();
  loopsPerSecond  = loopCount - lastLoopCount;
  loopSpeed       = (float)1e6 / loopsPerSecond;
  lastLoopCount   = loopCount;
}
// end serialDump()

//-----------------------------------------------------------------------------
uint32_t TimerOverFlow(uint32_t currentTime)
{
  return (currentTime + CORE_TICK_RATE*(1));  //the Core Tick Rate is 1ms
}

//-----------------------------------------------------------------------------
// 10-10-2013 W6CQZ
// Writes tuning word in f0 to AD9834 frequency register 0 and/or
// f1 to register 1.  To set one register only pass 0 to f0 or f1
// This wants the 28 bit tuning word!
// This has been modified to support the AD9850

void program_ab(unsigned long f0, unsigned long f1)  // Loads the desired frequency register with a 28 bit tuning word
{
  int flow,fhigh;
  if(f0>0)
  {
    double x_freq = ((double) ((long)((f0/ref_calc)*10))/10);  // Convert the 28 bit tuning word to frequency
    FR_0 = x_freq;  // Set emulated register 0 to frequency
  }
  if(f1>0)
  {
    double x_freq = ((double) ((long)((f1/ref_calc)*10))/10);  // Convert the 28 bit tuning word to frequency
    FR_1 = x_freq;  // Set emulated register 1 to frequency
  }
}

//=============================================================================

unsigned long getRX()  // get the desired RX tuning word
{

  return fcalc0;  // Returns last set value for DDS register 0 (RX)

}

boolean jt65TXStatus()  // JT 65 Transmit Status
{
  if(jt65TXOn) { return true; } else {return false;}
}

boolean jt9TXStatus()  // JT 9 Transmit Status
{
  if(jt9TXOn) { return true; } else {return false;}
}

void setFlip()  // Initialize the software flipflop for the DDS register loading
{
  flipflop = false;
}

boolean flip()  // toggle the DDS register flipflop
{
  if(flipflop)
  {
    flipflop=false;
    return true;
  }
  else
  {
    flipflop=true;
    return false;
  }
}

boolean jt65FrameStatus()  // Check the JT65 Frame Status
{
  int i;
  boolean v = true;
  for(i=0; i<126; i++)
  {
    if((fsk65Vals[i] < 37581152) || (fsk65Vals[i] > 77041361))
    {
      v = false;
      break;
    }
  }
  return v;
}

boolean jt9FrameStatus()  // Check the JT9 Frame Status
{
  int i;
  boolean v = true;
  for(i=0; i<86; i++)
  {
    if((fsk65Vals[i] < 37581152) || (fsk65Vals[i] > 77041361))
    {
      v = false;
      break;
    }
  }
  return v;
}

void stx65(boolean v)  // Enable JT65 TX
{
  if(v) { jt65TXOn = true; } else { jt65TXOn = false; }
}

void stx9(boolean v)  // Enable JT9 TX
{
  if(v) { jt9TXOn = true; } else { jt9TXOn = false; }
}

/*
cmdMessenger command callback processors by W6CQZ
Defines the CAT command set and routines for Rebel
*/

void OnUnknownCommand()  // Do nothing if command is unknown
{
  // Do nothing - one of my all time favorites! \0/
}

void onGVersion()  // Return the firmware version
{
  // Command ID = 2;
  cmdMessenger.sendCmd(kAck,ROMVERSION);
  // Setup the Nokia Display
  glcd.clrScr();
  glcd.print("RX:",0,0);
  glcd.print("RXO:",0,8);
  glcd.print("TXO:",0,16); 
  glcd.print("DDS:",0,24);
  glcd.print("Batt:",0,32);
  glcd.print("HFWST Online",0,40);
}

void onGDDSVer()  // Return the DDS version
{
  // Command ID = 3;
  cmdMessenger.sendCmd(kAck,"AD9834");
}

void onGDDSRef()  // Return the DDS Reference frequency
{
  // Command ID = 4;
  cmdMessenger.sendCmd(kAck,Reference);
}

void onSLockPanel()  // Lock out the Rebel front panel controls
{
  // Command ID 5
  cmdMessenger.sendCmd(kAck,5);
}

void onSUnlockPanel()  // Unlock the Rebel front panel controls
{
  // Command ID 6
  cmdMessenger.sendCmd(kAck,6);
}

void onLoopSpeed()  // Return the loop speed
{
  // Command ID=7;
  cmdMessenger.sendCmd(kAck,loopSpeed);
}

void onGRXOffset()  // Return the RX Offset
{
  // Command ID=8;
  // Reads RX offset
  cmdMessenger.sendCmd(kAck,rxOffset);
  glcd.printNumF(rxOffset,0,25,8);
}

void onGTXOffset()  // Return the TX Offset
{
  // Command ID=9;
  // Reads TX offset
  cmdMessenger.sendCmd(kAck,txOffset);
  glcd.printNumI(txOffset,25,16);
}

void onSRXOffset()  // Set the RX Offset
{
  // Command ID=10,rx_offset_hz;
  // Sets the INTEGER value to offset RX
  int i = cmdMessenger.readIntArg();
  rxOffset = i;
  cmdMessenger.sendCmd(kAck,i);
}

void onSTXOffset()  // Set the TX Offset
{
  // Command ID=11,tx_offset_hz;
  // Sets the INTEGER value to offset TX
  int i = cmdMessenger.readIntArg();
  txOffset = i;
  cmdMessenger.sendCmd(kAck,i);
}

void onGBand()  // Return the band - Hardcoded to 40m
{
  // Command ID = 12;
  cmdMessenger.sendCmd(kAck,40);
}

void onGRXFreq()  // Return the RX frequency
{
  // Command ID = 13;
  cmdMessenger.sendCmd(kAck,fcalc0);

  float Xfreq = (-1 * (fcalc0/(268.435456e6 / Reference)) + (rxOffset * 2) + IF);
  glcd.printNumF(Xfreq,0,25,24); // True Freq of DDS (typical offset 700 Hz)
  dds_RX = Xfreq;
  float Xfreq1 = Xfreq - rxOffset;
  glcd.printNumF(Xfreq1,0,25,0); // Displayed Freq of DDS without offset
  float voltage = analogRead(Battery_pin);
  voltage = (voltage * conversion) + volt_drop;
  glcd.printNumF(voltage,2,30,32); // Display the Battery voltage

}
  
void onSRXFreq()  // Set the RX frequency
{
  // Command ID = 14,value;
  unsigned long frx = cmdMessenger.readIntArg();
  // calling my routine to take a direct tuning word
  // this sets only register 0 (RX) to its LO value
  // it ***DOES NOT*** account for the needed offset
  // to RX due to 2nd LO being shifted to give a CW
  // beat note.
  program_ab(frx, 0);  // Load the DDS Register 0
  dds.setFrequency(FR_0);  // Set the DDS to RX frequency
  cmdMessenger.sendCmd(kAck,frx);
  fcalc0=frx;
}

void onGTXStatus()  // Return the TX Status
{
  // Command ID 15
  if(txstat) 
  {
    cmdMessenger.sendCmd(kAck,1); 
  } else {
    cmdMessenger.sendCmd(kAck,0); 
  }
}

void onSTXOn()  // Set the TX Status
{
  // Command ID 16;
  // Clear any symoffset as this is an on time TX
  symoffset = 0;
  if(jt65Valid)
  {
    jt65TXOn = true;
    cmdMessenger.sendCmd(kAck,16);
  } else
  {
    jt65TXOn = false;
    cmdMessenger.sendCmd(kError,16);
  }    
}

void onS9TXOn()  // Return the JT9 TX status
{
  // Command ID 17;
  // Clear any symoffset as this is an on time TX
  symoffset = 0;
  if(jt9Valid)
  {
    jt9TXOn = true;
    cmdMessenger.sendCmd(kAck,17);
  } else
  {
    jt9TXOn = false;
    cmdMessenger.sendCmd(kError,17);
  }    
}

void onSTXOff()  // Turn off TX
{
  // Command ID 18;
  jt65TXOn = false;
  jt9TXOn = false;
  cmdMessenger.sendCmd(kAck,18);
}
void onSDTXOn()
{
  // Command ID 19;
  long i = cmdMessenger.readIntArg();
  if(i>0 & i <45)
  {
    symoffset = i;
    if(jt65Valid)
    {
      jt65TXOn = true;
      cmdMessenger.sendCmdStart(kAck);
      cmdMessenger.sendCmdArg(19);
      cmdMessenger.sendCmdArg(i);
      cmdMessenger.sendCmdEnd();
    } else
    {
      jt65TXOn = false;
      cmdMessenger.sendCmdStart(kError);
      cmdMessenger.sendCmdArg(19);
      cmdMessenger.sendCmdArg(-1);
      cmdMessenger.sendCmdEnd();
    }
  } else
  {
    jt65TXOn = false;
    cmdMessenger.sendCmdStart(kError);
    cmdMessenger.sendCmdArg(19);
    cmdMessenger.sendCmdArg(i);
    cmdMessenger.sendCmdEnd();
  }
}

void onSD9TXOn()  // Return the JT9 TX status
{
  // Command ID 20;
  long i = cmdMessenger.readIntArg();
  if(i>0 & i <45)
  {
    symoffset = i;
    if(jt9Valid)
    {
      jt9TXOn = true;
      cmdMessenger.sendCmdStart(kAck);
      cmdMessenger.sendCmdArg(20);
      cmdMessenger.sendCmdArg(i);
      cmdMessenger.sendCmdEnd();
    } else
    {
      jt9TXOn = false;
      cmdMessenger.sendCmdStart(kError);
      cmdMessenger.sendCmdArg(20);
      cmdMessenger.sendCmdArg(-1);
      cmdMessenger.sendCmdEnd();
    }
  } else
  {
    jt9TXOn = false;
    cmdMessenger.sendCmdStart(kError);
    cmdMessenger.sendCmdArg(20);
    cmdMessenger.sendCmdArg(i);
    cmdMessenger.sendCmdEnd();
  }
}

void onDoCWID()  // Send the CW Id
{
  // Command ID 21;
  // Expects CWID as string and TX QRG as integer tuning word
  // *** SENDS CWID **** Upon receipt of this command as in N O W
  unsigned long cwidqrg = 0;  // DDS tuning word for CWID QRG
  String cwid = cmdMessenger.readStringArg();
  cwid.trim();
  cwidqrg = cmdMessenger.readIntArg();
  boolean txcwid = false;
  if(cwid.length()>2) { txcwid = true; } else { txcwid = false; }
  if(cwidqrg >= 37581152 & cwidqrg <= 77041361 & txcwid) { txcwid = true; } else { txcwid = false; } 
  
  if(txcwid)
  {
    LEDMorseSender cqSender(TX_OUT);  // Morse code TX handler
    cqSender.setup(); // Setup Morse TX handler
    // OK this should set the TX QRG to the sync +/- a little (I'm sending a tuning word with the command so that's done in HFWST)
    // The message to send is in (string)cwid and tuning word in cwidqrg
    cqSender.setMessage(cwid);
    cqSender.setWPM(25.0);
    // DDS register 1 (B) is by default used for TX - no need to preserve it right now
    program_ab(0, cwidqrg); // Remember - program_ab takes a tuning word and sets A and B registers if value > 0.  If = 0 it skips that register.
    delay(10);
    dds.setFrequency(FR_1);
//    digitalWrite(FREQ_REGISTER_BIT, HIGH);   // FR One is selected
    cqSender.sendBlocking();  // I set the default to 25 WPM so just calling the sender without any parameters
    dds.setFrequency(dds_RX);
//    digitalWrite(FREQ_REGISTER_BIT, LOW);   // FR Zero is selected
    cmdMessenger.sendCmdStart(kAck);
    cmdMessenger.sendCmdArg(cwid);
    cmdMessenger.sendCmdArg(cwidqrg);
    cmdMessenger.sendCmdEnd();
  }
  else
  {
    // Failed - due to string format or DDS value - flip a coin.
    cmdMessenger.sendCmdStart(kError);
    cmdMessenger.sendCmdArg(cwid);
    cmdMessenger.sendCmdArg(cwidqrg);
    cmdMessenger.sendCmdEnd();
  }
}

void onGClearTX()  // Clear TX frame
{
  // Command ID 22;
  // Clears fskVals[] and sets jtTXValid false
  jt65Valid = false;
  jt9Valid = false;
  int i;
  for(i=0; i<129; i++) { fsk65Vals[i]=0; }
  for(i=0; i<88; i++) { fsk9Vals[i]=0; }
  cmdMessenger.sendCmd(kAck,22);
}

void onSTXFreq()  // Prep the FSK values
{
  // Command ID 23;
  // Prepares FSK value receiver.  Once this is acked you can
  // begin uploading FSK values
  if(jt9TXOn || jt65TXOn)
  {
    cmdMessenger.sendCmd(kError,23);
  }
  else
  {
    cmdMessenger.sendCmd(kAck,23);
    jt65Valid = false;
    jt9Valid = false;
  }
}

void onGLoadTXBlock()  // Load the JT65 DDS tuning words
{
  // Command ID=24,Block {1..32},I1,I2,I3,I4;
  // Loading 4 Integer tuning words from Block # to Block #+3
  // into master TX array fskVals[0..127]
  // It is perfectly fine to send same block twice - this
  // allows correction should a block be corrupted in transit.
  // Modifying this to take the full 126 value frame so I don't
  // have to shuffle things here.  To keep it simple using 128
  // elements with extra 2 set to 0.  Just keeps the 4 value
  // chunk idea working in easy mode.  TX Routine will only
  // go 126 :)
  int i = 0;
  int block = 0;
  unsigned long i1 = 0;
  unsigned long i2 = 0;
  unsigned long i3 = 0;
  unsigned long i4 = 0;
  block = cmdMessenger.readIntArg();
  i1 = cmdMessenger.readIntArg();
  i2 = cmdMessenger.readIntArg();
  i3 = cmdMessenger.readIntArg();
  i4 = cmdMessenger.readIntArg();
  
  if(block<1)
  {
    jt65Valid = false;
    cmdMessenger.sendCmdStart(kError);  // NAK
    cmdMessenger.sendCmdArg(24);        // Command
    cmdMessenger.sendCmdArg(block);     // Parameter count where it went wrong
    cmdMessenger.sendCmdEnd();
  }
  else
  {
    // Have right value count - (eventually) validate (now) just stuff them into values array.
    //if(block>0) {i=block*4;} else {i=block;} // can skip this if mult by 0 is not an issue and just do i=block*4
    i=(block-1)*4; // This adjusts block to be 0...31 - I need to spec 1...32 above to be sure I'm reading a value
    // since cmdMessenger sets an Int value to 0 if it's not present.
    fsk65Vals[i]=i1;
    i++;
    fsk65Vals[i]=i2;
    i++;
    fsk65Vals[i]=i3;
    i++;
    fsk65Vals[i]=i4;
    // Echo the block back for confirmation on host side.
    cmdMessenger.sendCmdStart(24); // This was last block and simple range check = all good.
    cmdMessenger.sendCmdArg(block);
    cmdMessenger.sendCmdArg(i1);  // Echo the values back just to be sure.
    cmdMessenger.sendCmdArg(i2);  // After all... this is a TX routine so
    cmdMessenger.sendCmdArg(i3);  // it really is a good idea to double
    cmdMessenger.sendCmdArg(i4);  // check the values got in correct.
    cmdMessenger.sendCmdEnd();
    if(block==32) { jt65Valid = true; } else { jt65Valid = false; }
  }
}

void onG9LoadTXBlock()  // Load the JT9 DDS tuning words
{
  // Command ID=25,Block {1..22},I1,I2,I3,I4;
  // Loading 4 Integer tuning words from Block # to Block #+3
  // into master TX array fsk9Vals[0..87]
  // It is perfectly fine to send same block twice - this
  // allows correction should a block be corrupted in transit.
  // Modifying this to take the full 85 value frame so I don't
  // have to shuffle things here.  To keep it simple using 88
  // elements with extra 3 set to 0.  Just keeps the 4 value
  // chunk idea working in easy mode.  TX Routine will only
  // go 85 :)
  int i = 0;
  int block = 0;
  unsigned long i1 = 0;
  unsigned long i2 = 0;
  unsigned long i3 = 0;
  unsigned long i4 = 0;
  block = cmdMessenger.readIntArg();
  i1 = cmdMessenger.readIntArg();
  i2 = cmdMessenger.readIntArg();
  i3 = cmdMessenger.readIntArg();
  i4 = cmdMessenger.readIntArg();
  
  if(block<1)
  {
    jt9Valid = false;
    cmdMessenger.sendCmdStart(kError);  // NAK
    cmdMessenger.sendCmdArg(25);        // Command
    cmdMessenger.sendCmdArg(block);     // Parameter count where it went wrong
    cmdMessenger.sendCmdEnd();
  }
  else
  {
    // Have right value count - (eventually) validate (now) just stuff them into values array.
    //if(block>0) {i=block*4;} else {i=block;} // can skip this if mult by 0 is not an issue and just do i=block*4
    i=(block-1)*4; // This adjusts block to be 0...21 - I need to spec 1...22 above to be sure I'm reading a value
    // since cmdMessenger sets an Int value to 0 if it's not present.
    fsk9Vals[i]=i1;
    i++;
    fsk9Vals[i]=i2;
    i++;
    fsk9Vals[i]=i3;
    i++;
    fsk9Vals[i]=i4;
    // Echo the block back for confirmation on host side.
    cmdMessenger.sendCmdStart(25); // This was last block and simple range check = all good.
    cmdMessenger.sendCmdArg(block);
    cmdMessenger.sendCmdArg(i1);  // Echo the values back just to be sure.
    cmdMessenger.sendCmdArg(i2);  // After all... this is a TX routine so
    cmdMessenger.sendCmdArg(i3);  // it really is a good idea to double
    cmdMessenger.sendCmdArg(i4);  // check the values got in correct.
    cmdMessenger.sendCmdEnd();
    if(block==22) { jt9Valid = true; } else { jt9Valid = false; }
  }
}

void onGFSKVals()  // Return the JT65 DDS tuning words
{
  // Command ID=26;
  // Dumps 126 FSK values previously uploaded
  // for a JT65 frame.
  int i=0;
  cmdMessenger.sendCmdStart(kAck);
  cmdMessenger.sendCmdArg("FSK VALUES FOLLOW");
  for(i=0; i<126; i++)
  {
    cmdMessenger.sendCmdArg(fsk65Vals[i]);
  }
  cmdMessenger.sendCmdEnd();
}

void onG9FSKVals()  // Return the JT9 DDS tuning words
{
  // Command ID=27;
  // Dumps 88 FSK values previously uploaded
  // for a JT9 frame.
  int i=0;
  cmdMessenger.sendCmdStart(kAck);
  cmdMessenger.sendCmdArg("FSK VALUES FOLLOW");
  for(i=0; i<88; i++)
  {
    cmdMessenger.sendCmdArg(fsk9Vals[i]);
  }
  cmdMessenger.sendCmdEnd();
}

void onGGPSGrid()  // Returns the GPS Grid Square - not implemented
{
  // Command ID=28;
  // Returns GPS derived grid
  cmdMessenger.sendCmd(kAck,28);
}

void onGGPSTime()  // Returns the GPS time - not implemented
{
  // Command ID=29;
  // Returns GPS Time
  cmdMessenger.sendCmd(kAck,29);
}


