/*
DDS VFO - Tunes from 1 to 30 MHz with programmable step speed
Author: Glen Popiel - KW5GP

Uses M0XPD DDS Library by Paul Darlington, M0XPD
*/

// include the M0XPD DDS Library:
#include <DDS.h>
// Include the Nokia 5110 Library
#include <LCD5110_Basic.h>
// Include the Rotary Encoder Library
#include <Encoder.h>

//======================================
// AD9850 Module....
// set pin numbers:
const int W_CLK = 13;
const int FQ_UD = 10; 
const int DATA = 11;
const int RESET = 9;

// Nokia 5110 LCD Module
// Set Pin Numbers
const int CLK = 8;
const int DIN = 7;
const int DC = 6;
const int RST = 4;
const int CE = 5;

const int encoder_PinA = 2;  // Pin A of the Rotary Encoder
const int encoder_PinB = 3;  // Pin B of the Rotary Encoder
const int encoder_Switch_Pin = A4;  // Encoder Pushbutton Switch input pin

double step_value[] = {1,10,100,1000,10000,100000,1000000};

const int debounceInterval = 500;  //disable step change for 500 ms between encoder pushbutton press
boolean encoder_Switch;  
int current_step = 0;  // Default Tuning Step is 1 Hz
int encoder_Pos = 0;  
int old_encoder_Pos = 0;

const double range_low = 1000000; // Low end of range (1 MHz)
const double range_high = 30000000; // High end of range (30 MHz)
unsigned long int delay_time = 2000;

double start_freq = 7000000;  // Starting freq

double dds_freq = start_freq;

// Instantiate the DDS
DDS dds(W_CLK, FQ_UD, DATA, RESET);

// Instantiate the LCD
LCD5110 glcd(CLK,DIN,DC,RST,CE); //  Assign the Nokia 5110 LCD Pins

extern uint8_t SmallFont[];  // define the Nokia Font  

// Instantiate the Rotary Encoder
Encoder Enc(encoder_PinA, encoder_PinB);

void setup() 
{
  // Set up the Nokia 5110 Display
  glcd.InitLCD(60);  // Initialize 
  glcd.setFont(SmallFont);  // Use Small Font
  
  // Display the Startup screen
  glcd.clrScr();
  glcd.print("KW5GP", CENTER,0);
  glcd.print("DDS VFO",CENTER,8);
  glcd.print("1-30 MHz",CENTER,16);
  glcd.print("Initializing", CENTER,40);
  
  // start up the DDS...   
  dds.init();  
  // (Optional) trim if your xtal is not at 125MHz...
  dds.trim(125000322); // enter actual osc freq 
  
  delay(3000);
  glcd.clrScr();
  
  // Setup the LCD Display for operation
  glcd.print("Freq:",0,0);  // Display the Frequency information on the LCD
  glcd.print("Step:",0,24);
  
 // Set up the Rotary Encoder
  pinMode (encoder_PinA,INPUT);
  pinMode (encoder_PinB,INPUT);
  pinMode (encoder_Switch_Pin, INPUT);
  // Enable the Internal Pull-up resistor on the Encoder Inputs  
  digitalWrite(encoder_PinA, HIGH);
  digitalWrite(encoder_PinB, HIGH);
  digitalWrite(encoder_Switch_Pin, HIGH);

  // start the oscillator...
  set_DDS(dds_freq);

  glcd.printNumI(dds_freq,30,0,7);
  glcd.printNumI(step_value[current_step],30,24,7);
 
}  // End Setup Loop

void loop() 
{

read_encoder(); // Read the encoder and set the Tx/Rx frequencies
read_pushbutton();  // Read the Encoder Pushbutton switch and set Fast/Slow tuning mode

}  // End Main Loop

void read_encoder()
{
  // Read the Encoder  
  encoder_Pos = Enc.read()/4; // divide by 4 to match encoder detent
  if (encoder_Pos != old_encoder_Pos) // If the Encoder has changed update freq
  {
    if (encoder_Pos < old_encoder_Pos) // If we're increasing frequency
    {
      if (dds_freq >= range_high) // Limit to top end of band
      {
        dds_freq = range_high;
      } else {
        dds_freq = dds_freq + step_value[current_step];
      }
     
    } else {
      if (dds_freq <= range_low)// We're decreasing frequency, limit to low end of band
      {
        dds_freq = range_low;
      } else {
        dds_freq = dds_freq - step_value[current_step];
      }
    }

//    dds.setFrequency(tx_freq); // Set the TX frequency
    old_encoder_Pos = encoder_Pos;
    set_DDS(dds_freq);  
    glcd.printNumI(dds_freq,30,0,7);  // display the Tx Freq
    
  }
}  

void read_pushbutton()
{
  // Read the Encoder Pushbutton Switch
  encoder_Switch = digitalRead(encoder_Switch_Pin);
  if(encoder_Switch == LOW && millis() > delay_time) // Check to see if pressed
  {
    // if it's changed, toggle the step size but don't allow again for debounce time
    delay_time = millis() + debounceInterval; // if it's changed, toggle the step size but don't allow again for debounce time
    current_step = current_step + 1; // Change the Step pointer
    if (current_step >6)
    {
      current_step = 0;
    }

    glcd.printNumI(step_value[current_step],35,24,7);
  }
}  

void set_DDS(double set_freq)
{
  dds.setFrequency(set_freq);  // Set the DDS to the starting freq
  if (set_freq >= range_high)
  {
    set_freq = range_high;
  }
  if (set_freq <= range_low)
  {
    set_freq = range_low;
  }
}

