/* 

Yeasu G-450A Rotor Controller
Replaces Yeasu Rotor Analog Controller Board
Written by Glen Popiel, KW5GP

*/

#define debug_mode 1

#include <Wire.h>  // I2C Library
#include <EEPROM.h>  // EEPROM Library
#include "ADS1115.h" // Library Updated to fix errors
#include "I2Cdev.h"  // Library Updated to fix errors

ADS1115 adc; // Define as adc

#define dial_left 3  // Left (CCW) Dial Drive Pin
#define dial_right 5  // Right (CW) DIal Drive Pin
#define rotate_left 6  // Left (CCW) Rotor Drive Pin
#define rotate_right 7  // Right (CW) Rotor Drive Pin
#define overlap_led 2  // Overlap LED Pin
#define dial_enable 4  // Dial Drive Enable Pin
#define move_sense 8  // Rotor Movement Sense Pin


int CWDialSpeed = 150; // PWM setting for Clockwise Dial Speed
int CCWDialSpeed = 148; // PWM setting for Counterclockwise Dial Speed

#define BAUD_RATE 9600

#define EEPROM_ID_BYTE 1   // EEPROM ID to validate EEPROM data location
#define EEPROM_ID  56  // EEPROM ID Value
#define EEPROM_AZ_CAL_0 2    // Azimuth Zero Calibration EEPROM location     
#define EEPROM_AZ_CAL_MAX 4  // Azimuth Max Calibration Data EEPROM location  (450 degrees)
#define EEPROM_DIAL_CAL_0 6    // Dial Zero Calibration Data EEPROM location  
#define EEPROM_DIAL_CAL_MAX 8  // Dial Max Calibration Data EEPROM location  (450 degrees)

#define AZ_CAL_0_DEFAULT 295  // Rotor Zero Calibration Point Default Value
#define AZ_CAL_MAX_DEFAULT 25350  // Rotor Max Calibration Point Default Value
#define DIAL_CAL_0_DEFAULT 300  // Dial Motor Zero Calibration Point Default Value
#define DIAL_CAL_MAX_DEFAULT 25350  // Dial Motor Max Calibration Point Default Value

#define AZ_Tolerance 1  // Rotor Positioning Tolerance in Degrees
#define DIAL_Tolerance 2  // Dial Motor Positioning Tolerance in Degrees

//variables
byte inByte = 0;  // incoming serial byte
byte serial_buffer[50];  // incoming serial byte buffer
int serial_buffer_index = 0;  // Pointer to the character in the serial buffer
int set_AZ;  // Azimuth set value          
int set_DIAL;  // Dial set value
int current_AZ;  // Current Azimuth raw value
int current_DIAL;  // Current Dial raw value
String Serial_Send_Data; // Data to send to Serial Port
int AZ_0;  // Azimuth Zero Value from EEPROM
int AZ_MAX; // Azimuth Max Value from EEPROM
int DIAL_0;  // Dial 0 Value from EEPROM
int DIAL_MAX;  //Dial Max Value from EEPROM
int AZ_Degrees; // mapped AZ ADC value to Degrees
int DIAL_Degrees; // mapped DIAL ADC value to Degrees
String Requested_AZ; // RS232 Requested Azimuth - M and short W command
String Requested_DIAL; //RS232 Requested Azimuth and Dial - Full W command
int AZ_To; // Requested AZ Move 
int DIAL_To; // Requested DIAL Move
int AZ_Distance; // Distance to move AZ
int DIAL_Distance; // Distance to move DIAL
int overlap_status;  // indicates we're past 360 Degrees
long manual_move;  // manual move flag

int map_az_0, map_az_max, map_dial_0, map_dial_max, map_az, map_dial;  // variables to map A/D Values to Degrees

void setup() 
{
  pinMode(dial_right, OUTPUT);  // Set the pin modes
  pinMode(dial_left, OUTPUT);
  pinMode(rotate_left, OUTPUT);
  pinMode(rotate_right, OUTPUT);
  pinMode(overlap_led, OUTPUT);
  
  digitalWrite(move_sense, HIGH);  // Turn on the pullup resistor
  digitalWrite(dial_right, LOW);  // Turn off the motor drive
  digitalWrite(dial_left, LOW);
  digitalWrite(rotate_left, LOW);
  digitalWrite(rotate_right, LOW);
  
  Serial.begin(BAUD_RATE); // initialize serial communication 
  
  if (debug_mode) {Serial.println("Start Initializing");}

  Wire.begin();  // join I2C bus
  if (debug_mode) {Serial.println("Join I2C bus");}
  adc.initialize(); // initialize ADS1115 16 bit A/D chip
  if (debug_mode) {Serial.println("Initialize ADS1115");}
  Wire.beginTransmission(0x48); // Begin ADC setup
  if (debug_mode) {Serial.println("Begin ADC Setup");}
  Wire.write(0x1);
  //  Connect to adc and send two bytes - Set Config Reg to all Ones
  if (debug_mode) {Serial.println("Write ADC config reg");}
  Wire.write(0x7F);  //  MSB 
  if (debug_mode) {Serial.println("Write ADC 0x7f");}
  Wire.write(0xFF);    //  LSB
  if (debug_mode) {Serial.println("Write ADC 0xFF");}
  Wire.endTransmission();
  
  if (debug_mode) {Serial.println("ADC Initialized");}
  
  adc.setMode(ADS1115_MODE_CONTINUOUS); // free running conversion
  adc.setGain(ADS1115_PGA_6P144);  // set adc gain 6.144 range, .0001875 volts/step
  adc.setRate(ADS1115_RATE_475);  // set adc sample rate to 475 samples per second
  adc.setMultiplexer(ADS1115_MUX_P0_NG);  // AN0+ Vs ground 

  set_AZ = -1;  // Set rotor movement off
  set_DIAL = -1;  // Set dial movement off
  
  read_eeprom_cal_data();  // Read the calibration data from EEPROM

  read_adc();  // Read the A/D converter
  
  // Map AZ to degrees
  if (debug_mode) {Serial.println(current_AZ);}
  
  AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 450);  // Map the A/D Value to calibrated degrees
  
  if (AZ_Degrees >360) {    // Turn on overlap LED if over 360 degrees
    overlap_led_on();
    overlap_status = 1;
  } else {
    overlap_led_off();  // Otherwise turn off overlap LED
    overlap_status = 0;
  }  
  
  // Add dial sync at startup here
  if (debug_mode) {Serial.println("**** Dial init start *******");}

  read_adc();  // Read the current rotor Azimuth
  set_DIAL = AZ_Degrees;  // Set the dial position to current Azimuth
  check_move();  // Move the dial to match Azimith

  if (debug_mode) {Serial.println("**** Dial init complete *******");}
  
}  // End Setup Loop

void loop() 
{
  check_serial();  // Check for Serial Data
  check_move();  // Manage Rotor and Dial Movement
  
  if (digitalRead(move_sense) == 0 && set_AZ == -1)  // Check for a manual move
  {
    read_adc();  // Read the current azimuth 
    set_DIAL = AZ_Degrees;  // Set the dial to track the manual move
  
    if (debug_mode)
    {
      Serial.println("Manual Move sensed - Set_AZ = -1"); Serial.print("AZ_Degrees = "); Serial.println(AZ_Degrees);
    }
    check_move();  // Manage Dial movement  
    manual_move = millis()+2000;
  }
  if (manual_move > millis())
  {
    read_adc();
    AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 450);  // Map the Rotor A/D Value to Calibrated degrees
    // Give the dial time to settle
    set_DIAL = AZ_Degrees;
    if (set_DIAL <0)
    {
      set_DIAL = 0;
    }
    check_move();
    if (debug_mode)
    {
      Serial.println("Dial Settling"); Serial.print("AZ_Degrees = "); Serial.println(AZ_Degrees);
    }
  }
  
// End Main Loop

}  // End Main Loop

// Functions

void read_eeprom_cal_data()  // Read the Calibration Data from EEPROM
{
  if (EEPROM.read(EEPROM_ID_BYTE) == EEPROM_ID) 
  {
    if (debug_mode) // Print the Calibration Data in Debug Mode
    {
      Serial.println("Read EEPROM Calibration Data Valid ID");
      Serial.print("AZ_CAL_0 = ");
      Serial.println((EEPROM.read(EEPROM_AZ_CAL_0) * 256) + EEPROM.read(EEPROM_AZ_CAL_0 + 1),DEC);
      Serial.print("AZ_CAL_MAX = ");      
      Serial.println((EEPROM.read(EEPROM_AZ_CAL_MAX) * 256) + EEPROM.read(EEPROM_AZ_CAL_MAX + 1),DEC);  
      Serial.print("DIAL_CAL_0 = ");    
      Serial.println((EEPROM.read(EEPROM_DIAL_CAL_0) * 256) + EEPROM.read(EEPROM_DIAL_CAL_0 +1 ),DEC);
      Serial.print("DIAL_CAL_MAX = ");      
      Serial.println((EEPROM.read(EEPROM_DIAL_CAL_MAX) *256) + EEPROM.read(EEPROM_DIAL_CAL_MAX + 1),DEC);
    }
    AZ_0 = (EEPROM.read(EEPROM_AZ_CAL_0)*256) + EEPROM.read(EEPROM_AZ_CAL_0 + 1);  // Rotor Zero Calibration Point Value
    AZ_MAX = (EEPROM.read(EEPROM_AZ_CAL_MAX)*256) + EEPROM.read(EEPROM_AZ_CAL_MAX + 1);  // Rotor Max Calibration Point Value
    DIAL_0 = (EEPROM.read(EEPROM_DIAL_CAL_0)*256) + EEPROM.read(EEPROM_DIAL_CAL_0 + 1);  // Dial Zero Calibration Point Value
    DIAL_MAX = (EEPROM.read(EEPROM_DIAL_CAL_MAX)*256) + EEPROM.read(EEPROM_DIAL_CAL_MAX + 1);  // Dial Max Calibration Point Value
    
  } else {  // initialize eeprom to default values
  
  if (debug_mode) 
  {
    Serial.println("Read EEPROM Calibration Data Invalid ID - setting to defaults");
  }
  // If no valid EEPROM Data Set the Calibration Values to Default and save to EEPROM
  AZ_0 = AZ_CAL_0_DEFAULT; 
  AZ_MAX = AZ_CAL_MAX_DEFAULT;
  DIAL_0 = DIAL_CAL_0_DEFAULT;
  DIAL_MAX = DIAL_CAL_MAX_DEFAULT;
  write_eeprom_cal_data();
  }
}

void write_eeprom_cal_data() // Write the Calibration Values to EEPROM
{
  if (debug_mode) {
    Serial.println("Writing EEPROM Calibration Data");
  }
  
  EEPROM.write(EEPROM_ID_BYTE,EEPROM_ID);  // Write the EEPROM ID Byte used to validate EEPROM data
  EEPROM.write(EEPROM_AZ_CAL_0,highByte(AZ_0));  // Write the Rotor Zero Calibration Value High Order Byte
  EEPROM.write(EEPROM_AZ_CAL_0 + 1,lowByte(AZ_0));  // Write the Rotor Zero Calibration Value Low Order Byte
  EEPROM.write(EEPROM_AZ_CAL_MAX,highByte(AZ_MAX));  // Write the Rotor Max Calibration Value High Order Byte
  EEPROM.write(EEPROM_AZ_CAL_MAX + 1,lowByte(AZ_MAX));  // Write the Rotor Max Calibration Value Low Order Byte
  EEPROM.write(EEPROM_DIAL_CAL_0,highByte(DIAL_0));  // Write the Dial Motor Zero Calibration Value High Order Byte
  EEPROM.write(EEPROM_DIAL_CAL_0 + 1,lowByte(DIAL_0));  // Write the Dial Motor Zero Calibration Value Low Order Byte
  EEPROM.write(EEPROM_DIAL_CAL_MAX,highByte(DIAL_MAX));  // Write the Dial Motor Max Calibration Value High Order Byte
  EEPROM.write(EEPROM_DIAL_CAL_MAX + 1,lowByte(DIAL_MAX));  // Write the Dial Motor Max Calibration Value Low Order Byte
}

void check_serial() // Check the Serial port for incoming commands
{
  if (Serial.available() > 0) 
  {
    inByte = Serial.read();  // Read the serial character

    if (inByte == 10) {return;} // ignore Line Feeds
    if (inByte !=13) {  // Add to buffer if not CR
      serial_buffer[serial_buffer_index] = inByte;  // Add the character to the serial command buffer
             
      if (debug_mode) // Display the character received in debug mode
      {
        Serial.print("Received = ");
        Serial.println(serial_buffer[serial_buffer_index]);
      }
      
      serial_buffer_index++;  // increment the serial data buffer pointer
    
    } else {  // It's a CR, execute command

      if ((serial_buffer[0] > 96) && (serial_buffer[0] < 123))  // it's lowercase, convert to uppercase
      {
        serial_buffer[0] = serial_buffer[0] - 32;
      }

    
      switch (serial_buffer[0])   // Decode first character of command
      {
                 
        case 65:  // "A" Command - All Stop
      
        if (debug_mode) {Serial.println("A Command Received");}
        az_rotate_stop();
        break;
      
        case 66:  // "B" Command - Send current dial position
      
        if (debug_mode) {Serial.println("B Command Received");}
        send_current_dial();
        break;
      
        case 67:      // "C" command - return current azimuth
      
        if (debug_mode) {Serial.println("C Command Received");}
        if (debug_mode) {Serial.println(serial_buffer_index);}
        if ((serial_buffer_index == 2) & (serial_buffer[1] == 50)) 
        {
          if (debug_mode) {Serial.println("C2 Command Received");}        
          send_current_az_dial();  // Send the dial position if C2 command
        } else { 
          send_current_az();
        }
        break;
      
        case 68:  // "D" command - rotate dial left
      
        if (debug_mode) {Serial.println("D Command Received");}
        rotate_dial_left();
        break;
      
        case 69:  // "E" command - Stop Dial Rotation
      
        if (debug_mode) {Serial.println("E Command Received");}
        dial_rotate_stop();
        break;
              
        case 70:  // "F" command - Set full scale calibration value
      
        if (debug_mode) {Serial.println("F Command Received");}
        if ((serial_buffer_index == 2) & (serial_buffer[1] == 50)) 
        {
          if (debug_mode) {Serial.println("F2 Command Received");}        
          set_dial_max_cal();  // Set the dial full scale calibration value if F2 command
        } else { 
          set_az_max_cal();
        }
        break;
      
        case 76:  // "L" Command - Rotate Left (CCW)
      
        if (debug_mode) {Serial.println("L command Received");}
        rotate_az_ccw();
        break;
      
        case 77:  // "M" command - Rotate to Azimuth xxx degrees
      
        if (debug_mode) {Serial.println("M Command Received");}
        rotate_az_to();
        break;
            
        case 79:  // "O" command - Set zero calibration point value
      
        if (debug_mode) {Serial.println("O Command Received");}
        if ((serial_buffer_index == 2) & (serial_buffer[1] == 50)) 
        {
          if (debug_mode) {Serial.println("02 Command Received");}        
         set_dial_0_cal();  // Set Dial zero calibration point value if O2
        } else { 
          set_az_0_cal();
        }
        break;
      
        case 82:  // "R" command - Rotate Right (CW)
      
        if (debug_mode) {Serial.println("R Command Received");}
        rotate_az_cw();
        break;
      
        case 83:  // "S" command - Stop all rotation
      
        if (debug_mode) {Serial.println("S Command Received");}
        az_rotate_stop();
        dial_rotate_stop();
        break;
      
        case 85:  // "U" command - Rotate Dial Right (CW)
      
        if (debug_mode) {Serial.println("U Command Received");}
        rotate_dial_right();
        break;
            
        case 87:  // "W" command - Rotate Azimuth to xxx degrees
      
        if (debug_mode) {Serial.println("W Command Received");}
        rotate_az_to();
        break;
      
      }
      serial_buffer_index = 0;  // Clear the serial buffer
      serial_buffer[0] = 0;
    }
  }
}
  
void send_current_az() // Send the Current Azimuth
{
  read_adc();  // Read the A/D Converter
  // Map it to degrees
  if (debug_mode) {Serial.println(current_AZ);}
  map_az = current_AZ - AZ_0; // Subtract the Zero Offset calibration

  
  if (debug_mode)
  {
    Serial.print("map_az = "); Serial.println(map_az);
    Serial.print("AZ_MAX = "); Serial.println(AZ_MAX);
    Serial.print("AZ_0 = "); Serial.println(AZ_0);
  }
  
  AZ_Degrees = map(map_az, 0, AZ_MAX - AZ_0, 0, 450);  // Map the Azimuth value to calibrated degrees
  if (debug_mode) {Serial.println(AZ_Degrees);}
  if (AZ_Degrees <0) {AZ_Degrees = 0;}  // Set Azimuth to 0 if below Zero
  if (AZ_Degrees>450) {AZ_Degrees = 450;}  // Set the Azimuth to 450 if above 450  
  // Send it back via serial
  Serial_Send_Data = "";
  
  if (AZ_Degrees < 100)  // pad with 0's if needed
  {
    Serial_Send_Data = "0";
  }
  if (AZ_Degrees < 10) 
  {
    Serial_Send_Data = "00";
  }
  Serial_Send_Data = "+0" + Serial_Send_Data + String(AZ_Degrees);
  Serial.println(Serial_Send_Data);  // Return value via RS-232 port
}
  
void send_current_az_dial() // Send the current Rotor and Dial values in degrees
{
  read_adc();  // Read the A/D Converter
  // Map AZ to degrees
  if (debug_mode) {Serial.println(current_AZ);}
  AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 450);
  // Map Dial to degrees
  if (debug_mode) {Serial.println(current_DIAL);}
  DIAL_Degrees = map(current_DIAL, DIAL_0, DIAL_MAX, 0, 450);
  if (debug_mode) {Serial.println(DIAL_Degrees);} 
  if (debug_mode) {Serial.println(AZ_Degrees);}  
  // Send it back via serial
  Serial_Send_Data = "";
  if (AZ_Degrees < 100) // pad with 0's if needed
  { 
    Serial_Send_Data = "0";
  }
  if (AZ_Degrees < 10) 
  {
    Serial_Send_Data = "00";
  }
  Serial_Send_Data = "+0" + Serial_Send_Data + String(AZ_Degrees) + "+0";
  if (DIAL_Degrees < 100)  // pad with 0's if needed
  {
    Serial_Send_Data = Serial_Send_Data + "0";
  }
  if (DIAL_Degrees < 10) 
  {
    Serial_Send_Data = Serial_Send_Data+ "0";
  }
  Serial_Send_Data = Serial_Send_Data + String(DIAL_Degrees);
  Serial.println(Serial_Send_Data);  // Return value via RS-232 port  
}

void set_az_max_cal() // Set the Rotor Azimuth Max Calibration Value
{
  if (debug_mode) {Serial.println("Calibrate Azimuth Max Function");}    
  read_adc();  // Read the A/D Converter
  // save current az values to EEPROM - Zero Calibration
  if (debug_mode) {Serial.println(current_AZ);} 
  AZ_MAX = current_AZ;  // Update the Rotor Azimuth Max value calibration variable
  write_eeprom_cal_data();  // Write the calibration data to EEPROM
}  

void set_dial_max_cal() // Set the Dial Azimuth Max Calibration Value 
{
  if (debug_mode) {Serial.println("Calibrate Dial Max Function");}    
  read_adc();  // Read the A/D Converter
  // save current az values to EEPROM - Zero Calibration
  if (debug_mode) {Serial.println(current_DIAL);} 
  DIAL_MAX = current_DIAL;  // Update the Dial Azimuth Max value calibration variable
  write_eeprom_cal_data();  // Write the calibration data to EEPROM  
} 


void rotate_az_ccw() // Rotate Left (CCW)
{
  digitalWrite(rotate_left, HIGH);
  digitalWrite(rotate_right, LOW);  
}

void rotate_az_cw() // Rotate Right (CW)
{
  digitalWrite(rotate_right, HIGH);
  digitalWrite(rotate_left, LOW);
}

void rotate_dial_right() // Rotate Dial Right (CW)
{
  analogWrite(dial_right, CWDialSpeed);  // Rotate using PWM
  digitalWrite(dial_left, LOW);
  digitalWrite(dial_enable, HIGH);  // Enable Dial Rotation
}

void rotate_dial_left() // Rotate Dial Left (CCW) 
{
  analogWrite(dial_left, CCWDialSpeed);  // Rotate using PWM
  digitalWrite(dial_right, LOW); 
  digitalWrite(dial_enable, HIGH);   // Enable Dial Rotation
}

void az_rotate_stop() // Stop Rotation
{
  digitalWrite(rotate_right, LOW);
  digitalWrite(rotate_left, LOW);  
  set_AZ = -1;
}

void dial_rotate_stop() // Stop Dial Rotation
{
  digitalWrite(dial_right, LOW);
  digitalWrite(dial_left, LOW);  
  digitalWrite(dial_enable, LOW);
}

void overlap_led_on() // Turn on the Overlap LED
{
  if (overlap_status == 0) 
  {
    digitalWrite(overlap_led, HIGH);
    overlap_status = 1;
  }
}

void overlap_led_off()  // Turn off the Overlap LED
{
  if (overlap_status == 1) 
  {
    digitalWrite(overlap_led, LOW);
    overlap_status = 0;
  }
}

void send_current_dial() // Send the current Dial Value in degrees
{
  read_adc();  // Read the A/D Converter
  // Map it to degrees
  if (debug_mode) {Serial.println(current_DIAL);}
  DIAL_Degrees = map(current_DIAL, DIAL_0, DIAL_MAX, 0, 450);
  if (debug_mode) {Serial.println(DIAL_Degrees);} 
  // Send it back via serial
  Serial_Send_Data = "";
  if (DIAL_Degrees < 100)  // pad with 0's if needed
  {
    Serial_Send_Data = "0";
  }
  if (DIAL_Degrees < 10) 
  {
    Serial_Send_Data = "00";
  }
  Serial_Send_Data = "+0" + Serial_Send_Data + String(DIAL_Degrees);
  Serial.println(Serial_Send_Data);  // Return value via RS-232 port  
}

void rotate_az_to()  // Rotate to Azimuth xxx in degrees 
{
  if (debug_mode) {Serial.println("W or M Command -  Rotate Azimuth  To Function");}  
  // Decode Command - Format Wxxx/Mxxx where xxx = Azimuth to Move to 
  if (debug_mode) {Serial.println(serial_buffer_index);}
  if (serial_buffer_index == 4)  // Move Command is 4 characters Mxxx or Wxxx
  {
    if (debug_mode) {Serial.println("Value in [1] to [3]?");}
    Requested_AZ = (String(char(serial_buffer[1])) + String(char(serial_buffer[2])) + String(char(serial_buffer[3]))) ;
    AZ_To = (Requested_AZ.toInt()); // AZ Degrees to Move to as integer
    if (AZ_To <0) {AZ_To = 0;}  // Set it to 0 if below
    if (AZ_To >450) {AZ_To = 450;}  // set it to 450 if above
    
    if (debug_mode) {Serial.println(Requested_AZ);}
    if (debug_mode) {Serial.println(AZ_To);} 
      
    // set the move flag and start
    read_adc();  // Read the A/D Converter
    // Map it to degrees
    if (debug_mode) {Serial.println(current_AZ);}
    AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 450);  // Map the A/D value to calibrated degrees
    if (AZ_Degrees <= 0)
    {
      AZ_Degrees = 0;
    }
    if (AZ_Degrees >=450)
    {
      AZ_Degrees = 450;
    }
    if (AZ_Degrees >360)   // Turn on overlap led if over 450
    {  
      overlap_led_on();
    } else {
      overlap_led_off();
    }
    if (debug_mode) {Serial.println(AZ_Degrees);}
    AZ_Distance = AZ_To - AZ_Degrees;  // Calculate how far to move the rotor
    set_AZ = AZ_To;
    DIAL_Degrees = map(current_DIAL, DIAL_0, DIAL_MAX, 0, 450);
    if (debug_mode) {Serial.println(DIAL_Degrees);}
    DIAL_Distance = DIAL_To - DIAL_Degrees;  // Calculate how far to move the dial
    set_DIAL = AZ_Degrees;
    // Set Azimuth
    if (abs(AZ_Distance) <= AZ_Tolerance || (AZ_Degrees>=450 & AZ_Distance >0))  // No move needed
    {
      az_rotate_stop();  // Stop the rotation
      set_AZ = -1;
    } else {  // Move Azimuth - figure out which way
      if (AZ_Distance > 0)   //We need to move CW
      {
        rotate_az_ccw();
      } else {
        rotate_az_cw();
      }
    } 
  }
}

void set_az_0_cal() // Set the Rotor Zero Calibration Point value
{
  if (debug_mode) {Serial.println("Calibrate Azimuth Zero Function");}    
  read_adc();  // Read the A/D Converter
  // save current az value to EEPROM - Zero Calibration
  if (debug_mode) {Serial.println(current_AZ);} 
  AZ_0 = current_AZ;  // Save the current Azimuth in the zero calibration variable
  write_eeprom_cal_data();  // wite the calibration data to EEPROM
}

void set_dial_0_cal()  // Set the Dial Zero Calibration Point value
{
  if (debug_mode) {Serial.println("Calibrate Dial Zero Function");}    
  read_adc();  // Read the A/D Converter
  // save current dial value to EEPROM - Zero Calibration
  if (debug_mode) {Serial.println(current_DIAL);} 
  DIAL_0 = current_DIAL;  // Save the current dial Azimuth in the zero calibration variable
  write_eeprom_cal_data();  // wite the calibration data to EEPROM  
}


void read_adc() // Read the A/D Converter
{
 if (debug_mode) {Serial.println("Read ADC Function  ");}

 int RotorValue;
 adc.setRate(ADS1115_RATE_475); // Set the sample rate to 475 samples/sec
 adc.setGain(ADS1115_PGA_6P144); // Set the range to 0 to 6.144V
 adc.setMultiplexer(ADS1115_MUX_P0_NG);  // AN0+ Vs ground - Set Channel 0 to Single ended mode
 delay(10); // adc settling delay
 current_AZ = adc.getDiff0();  // Read A/D Channel 0 (Rotor)
 if (debug_mode) {Serial.println(current_AZ);}  
 adc.setMultiplexer(ADS1115_MUX_P1_NG);  // AN1+ Vs ground - Set Channel 1 to Single ended mode
 delay(10); // adc settling delay
 current_DIAL = adc.getDiff1();  // Read A/D Channel 1 (Dial)
 if (debug_mode) {Serial.println(current_DIAL);}  
}

void check_move() // Manage Rotor and Dial movement
{
  if (set_AZ != -1 || set_DIAL != -1) {   // We're moving - check and stop as needed
    read_adc();  // Read the A/D Converter
    // Map AZ to degrees
    if (debug_mode) {Serial.print("Current AZ: "); Serial.println(current_AZ);}
    AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 450);  // Map the Rotor A/D Value to Calibrated degrees
    if (AZ_Degrees >360)     // Turn on overlap led if over 360
    {    // Turn on overlap led
      overlap_led_on();
    } else {
      overlap_led_off();
    }
    // Map Dial to degrees
    if (debug_mode) {Serial.print("Current Dial: ");Serial.println(current_DIAL);}
    DIAL_Degrees = map(current_DIAL, DIAL_0, DIAL_MAX, 0, 450);  // Map the Dial A/D Value to Calibrated degrees
    if (debug_mode) {Serial.print("AZ_Degrees: "); Serial.println(AZ_Degrees);} 
    if (debug_mode) {Serial.print("Dial_Degrees: "); Serial.println(DIAL_Degrees);}  

    if (set_AZ != -1) // Calculate the rotor distance left to move if we're moving
    {
      AZ_Distance = set_AZ - AZ_Degrees;
      if (abs(AZ_Distance) <= AZ_Tolerance)  // We're there - No move needed
      {
        az_rotate_stop();
        set_AZ = -1;
      } else {  // Move Azimuth - figure out which way
        if (AZ_Distance > 0)   //We need to move CW
        {
          rotate_az_cw();
        } else {
          rotate_az_ccw();
        }    
      }
      set_DIAL = AZ_Degrees; 
   }
   if (set_DIAL != -1)  // Calculate the dial distance left to move if we're moving
   {
     DIAL_Distance = set_DIAL - DIAL_Degrees;
     if (abs(DIAL_Distance) <= DIAL_Tolerance)   // We're there - No move needed
     { 
       dial_rotate_stop();
       set_DIAL = -1;
      } else {  // Move Dial - figure out which way
        if (DIAL_Distance > 0)   //We need to move CW
        {
          rotate_dial_right();
         if (debug_mode) 
         {
          Serial.println("Need to move dial right");
          }
        } else {
          rotate_dial_left();
          if (debug_mode) 
          {
            Serial.println("Need to move dial left");
          }
        }    
      }   
    }
  } 
}  // End Loop()

