/* GPS-based Arduino Network Time Protocol Server

    by Glen Popiel - KW5GP

    based on NTP time server sketch posted by Ziggy2012 in the Arduino forum (forum.arduino.cc)

    Uses UIPEthernet library by Norbert Truchsess <norbert.truchsess@t-online.de>
    Uses TinyGPS library by Mikal Hart (www.http://arduiniana.org)
    Uses built-in SoftwareSerial library
    Uses LCD5110_Basic library by Henning Karlsen (http://www.rinkydinkelectronics.com)

*/

#define debug true  // Debug flag

#define NTP_PORT 123  // Define the NTP Port Number
#include <UIPEthernet.h>  // Include the UIPEthernet library
#include <TinyGPS.h>  // Include the TinyGPS Library
#include <SoftwareSerial.h>  // include the SoftwareSerial library so we can talk to the GPS module
#include <LCD5110_Basic.h>  // Include the Nokia LCD5110_Basic libary

#define gps_rxPin 6    // GPS Serial input pin (connects to GPS TxD)
#define gps_txPin -1    // GPS Serial output pin (connects to GPS RxD) Set to -1 (disabled) because we don't need to send anything to GPS
#define gps_reset_pin  4  //GPS Reset pin

// Nokia 5110 LCD Module pin numbers
#define CLK 2
#define DIN 3
#define DC 7
#define RST 9
#define CE 8

SoftwareSerial GPS(gps_rxPin, gps_txPin);  // Create a software serial port instance

EthernetUDP Udp;  // Create an EthernetUDP instance

byte mac[] = { 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF };  // Time Server MAC address

IPAddress ip(10, 242, 255, 95); // NTP Server IP Address

static const int NTP_PACKET_SIZE = 48;  // Set the NTP packet size

byte packetBuffer[NTP_PACKET_SIZE]; // allocate the buffers for receiving and sending data

TinyGPS gps;  // Create a GPS instance

int year; // Variable to hold the GPS year
byte month, day, hour, minute, second, hundredths;  // Variables to hold the GPS data
unsigned long age;  // Variable to hold the age of the GPS data
uint32_t timestamp, tempval;  // Variables used to calculate NTP time values
boolean gps_ok = false;  // gps data ok flag
unsigned long next_time; // Variables to hold the one second clock ticks derived from millis()
char x_date[10];  // Character array to hold the formatted date for display
char x_time[10];  // Character array to hold the formatted time for display

LCD5110 glcd(CLK, DIN, DC, RST, CE); // Create a Nokia 5110 LCD Display instance
extern uint8_t SmallFont[];  // define the Nokia Font

void setup ()
{
  // Set up the Nokia 5110 Display
  glcd.InitLCD(65);  // Initialize the LCD display, set the contract to 65
  glcd.setFont(SmallFont);  // Use Small Font

  // Display the Startup screen
  glcd.clrScr();
  glcd.print("KW5GP", CENTER, 8);
  glcd.print("GPS-based NTP", CENTER,16 );
  glcd.print("Time Server", CENTER, 24);

  // start the Ethernet and UDP:
  Ethernet.begin(mac, ip);
  Udp.begin(NTP_PORT);

#if debug
  Serial.begin(9600);
#endif

  pinMode(gps_rxPin, INPUT);// set up the serial I/O pins for the GPS module
  pinMode(gps_txPin, OUTPUT);
  pinMode(gps_reset_pin, OUTPUT);

  digitalWrite(gps_reset_pin, HIGH);  // Clear the GPS Reset

  GPS.begin(9600);  // Start the GPS serial port

  delay(3000);
  glcd.clrScr();  // Clear the LCD display
}

void loop ()
{
  if (!gps_ok)  // Wait for valid GPS data and update seconds display
  {
    glcd.print("GPS Starting", CENTER, 16);
    glcd.printNumI(millis() / 1000 , CENTER, 40);
  }

  getgps(); // Get the GPS data

  gps.crack_datetime(&year, &month, &day, &hour, &minute, &second, &hundredths, &age);  // Extract the GPS data into the variables

  if (age == TinyGPS::GPS_INVALID_AGE)  // Check to see if GPS data age is valid
  {
#if debug
    if (millis() > next_time)
    {
      Serial.print("Invalid GPS age  ");
      Serial.println (next_time);
      next_time = millis() + 1000;
    }
#endif

    gps_ok = false; // Set valid gps data flag to false as long as the age is invalid

  } else
  {
    if (!gps_ok)  // If the age is now valid but we haven't set the gps_ok flag
    {
      // First pass with valid GPS Data - Clear the LCD

#if debug
      Serial.println("GPS data now valid");
#endif

      glcd.clrScr();  // Clear ths LCD display
      gps_ok = true;  // Set the gps_ok flag
    }
    if (millis() > next_time) // Update the LCD display every second
    {
      if (year >= 2016) // The first few passes may give a bad year (2000) - wait for the year to become valid
      {
        sprintf(x_date, "%02d/%02d/%02d" , month, day, year - 2000);  // Format the GPS data into a character array for display
        sprintf(x_time, "%02d:%02d:%02d", hour, minute, second);

        glcd.print("NTP Server", CENTER, 0);  // Update the LCD display
        glcd.print("Date:", 0, 16);
        glcd.print(x_date, 30, 16);
        glcd.print("UTC :", 0, 24);
        glcd.print(x_time, 30, 24);


        next_time = millis() + 1000;  // Update the one second display update timer
      }
    }
  }

  if (gps_ok) // Prepare the NTP data packet if the GPS data is valid
  {
    processNTP(); // Convert the GPS data into an NTP data packet and check for NTP request
  }
}



// Function to convert the GPS data to an NTP packet and replay to NTP request
void processNTP()
{
  // Check for an incoming NTP data request
  int packetSize = Udp.parsePacket(); // Get the number of bytes in the received data packet
  if (packetSize) // If we've received a UDP data packet
  {
    Udp.read(packetBuffer, NTP_PACKET_SIZE);  // Read the packet into the packet buffer
    IPAddress Remote = Udp.remoteIP();  // Extract the sender's IP address
    int PortNum = Udp.remotePort(); // Extract the IP port the packet was received on

#if debug // Show the incoming packet data
    Serial.println();
    Serial.print("Received UDP packet size ");  // Display the packet length in bytes
    Serial.println(packetSize);
    Serial.print("From ");  // Display the Sender's IP address

    for (int i = 0; i < 4; i++) // Extact the IP address and format by octet
    {
      Serial.print(Remote[i], DEC);
      if (i < 3)
      {
        Serial.print(".");
      }
    }

    Serial.print(", port ");  // Display the IP port the packet was received on
    Serial.print(PortNum);

    byte LIVNMODE = packetBuffer[0];  // Display the Leap Second (LI) indicator, NTP Version, and Mode data of the incoming request
    Serial.print("  LI, Vers, Mode :");
    Serial.print(packetBuffer[0], HEX);

    byte STRATUM = packetBuffer[1]; // Display the NTP Stratum level of the incoming request
    Serial.print("  Stratum :");
    Serial.print(packetBuffer[1], HEX);

    byte POLLING = packetBuffer[2]; // Display the Polling interval of the incoming request
    Serial.print("  Polling :");
    Serial.print(packetBuffer[2], HEX);

    byte PRECISION = packetBuffer[3]; // Display the Precision value of the incoming request
    Serial.print("  Precision :");
    Serial.println(packetBuffer[3], HEX);

    /*  Uncomment to see incoming packet raw date - use with disgression - can gause loss of data from GPS
        for (int z = 0; z < NTP_PACKET_SIZE; z++) // Display the incoming packet Raw data
        {
          Serial.print(packetBuffer[z], HEX);
          if (((z + 1) % 4) == 0) {
            Serial.println();
          }
        }
        Serial.println();
    */

#endif

    // Prepare the response NTP packet
    packetBuffer[0] = 0b00100100;   // LI, Version, Mode
    packetBuffer[1] = 1 ;   // stratum
    packetBuffer[2] = 6 ;   // polling minimum
    packetBuffer[3] = 0xFA; // precision

    packetBuffer[7] = 0; // root delay
    packetBuffer[8] = 0;
    packetBuffer[9] = 8;
    packetBuffer[10] = 0;

    packetBuffer[11] = 0; // root dispersion
    packetBuffer[12] = 0;
    packetBuffer[13] = 0xC;
    packetBuffer[14] = 0;

    gps.crack_datetime(&year, &month, &day, &hour, &minute, &second, &hundredths, &age);  // Get the current GPS data

    timestamp =  numberOfSecondsSince1900Epoch(year, month, day, hour, minute, second); // Get the timestamp value (# of seconds since 1900)

#if debug
    Serial.println(timestamp);  // Print the timestamp and the date
    print_date(gps);
#endif

    tempval = timestamp;

    // Set the Reference ID
    packetBuffer[12] = 71; //"G";
    packetBuffer[13] = 80; //"P;
    packetBuffer[14] = 83; //"S";
    packetBuffer[15] = 0; //"0";

    // Build the reference timestamp
    packetBuffer[16] = (tempval >> 24) & 0XFF;
    tempval = timestamp;
    packetBuffer[17] = (tempval >> 16) & 0xFF;
    tempval = timestamp;
    packetBuffer[18] = (tempval >> 8) & 0xFF;
    tempval = timestamp;
    packetBuffer[19] = (tempval) & 0xFF;

    packetBuffer[20] = 0;
    packetBuffer[21] = 0;
    packetBuffer[22] = 0;
    packetBuffer[23] = 0;


    //copy the originate timestamp from the incoming UDP transmit timestamp
    packetBuffer[24] = packetBuffer[40];
    packetBuffer[25] = packetBuffer[41];
    packetBuffer[26] = packetBuffer[42];
    packetBuffer[27] = packetBuffer[43];
    packetBuffer[28] = packetBuffer[44];
    packetBuffer[29] = packetBuffer[45];
    packetBuffer[30] = packetBuffer[46];
    packetBuffer[31] = packetBuffer[47];

    //  Build the received timestamp
    packetBuffer[32] = (tempval >> 24) & 0XFF;
    tempval = timestamp;
    packetBuffer[33] = (tempval >> 16) & 0xFF;
    tempval = timestamp;
    packetBuffer[34] = (tempval >> 8) & 0xFF;
    tempval = timestamp;
    packetBuffer[35] = (tempval) & 0xFF;

    packetBuffer[36] = 0;
    packetBuffer[37] = 0;
    packetBuffer[38] = 0;
    packetBuffer[39] = 0;

    //  Build the transmit timestamp
    packetBuffer[40] = (tempval >> 24) & 0XFF;
    tempval = timestamp;
    packetBuffer[41] = (tempval >> 16) & 0xFF;
    tempval = timestamp;
    packetBuffer[42] = (tempval >> 8) & 0xFF;
    tempval = timestamp;
    packetBuffer[43] = (tempval) & 0xFF;

    packetBuffer[44] = 0;
    packetBuffer[45] = 0;
    packetBuffer[46] = 0;
    packetBuffer[47] = 0;

    // Reply to the IP address and port that sent the NTP request with the created NTP data packet
    Udp.beginPacket(Remote, PortNum);
    Udp.write(packetBuffer, NTP_PACKET_SIZE);
    Udp.endPacket();
  }
}

// Function to get the incoming GPS serial data
static bool getgps()
{

  while (GPS.available()) // Check for incoming serial GPS data
  {
    char c = GPS.read();  // Read the individual GPS character

#if debug
    //        Serial.print(c);// Diplay the incoming GPS data flowing - use with disgression, will cause TinyGPS to miss data
#endif

    if (gps.encode(c))  // Create a complete GPS sentence from the GPS serial data stream
    {
      return true;  // Return true if we have a complete GPS message
    }
  }
  return false; // Return false if there is no complete GPS message
}

// Constants needed for numberOfSecondsSince1900Epoch() function
const uint8_t daysInMonth [] PROGMEM = {
  31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
}; //must be const or the compiler complains

const unsigned long seventyYears = 2208988800UL; // to convert unix time to epoch

// Function to calculate the number of seconds since 1900/01/01 to calculate NTP timestamp
static unsigned long int numberOfSecondsSince1900Epoch(uint16_t y, uint8_t m, uint8_t d, uint8_t h, uint8_t mm, uint8_t s)
{
  if (y >= 1970)
    y -= 1970;
  uint16_t days = d;
  for (uint8_t i = 1; i < m; ++i)
    days += pgm_read_byte(daysInMonth + i - 1);
  if (m > 2 && y % 4 == 0)
    ++days;
  days += 365 * y + (y + 3) / 4 - 1;
  return days * 24L * 3600L + h * 3600L + mm * 60L + s + seventyYears;
}

// Only use the following functions in debug mode

#if debug

// Function to print the GPS date and time data
static void print_date(TinyGPS & gps)
{
  int year;
  byte month, day, hour, minute, second, hundredths;
  unsigned long age;
  gps.crack_datetime(&year, &month, &day, &hour, &minute, &second, &hundredths, &age);
  if (age == TinyGPS::GPS_INVALID_AGE)
    Serial.print(F("*******    *******    "));
  else
  {
    char sz[32];
    sprintf(sz, "%02d/%02d/%02d %02d:%02d:%02d :",
            month, day, year, hour, minute, second);
    Serial.print(sz);
  }
  print_int(age, TinyGPS::GPS_INVALID_AGE, 5);
}

// Function to print the age of the GPS data
static void print_int(unsigned long val, unsigned long invalid, int len)
{
  char sz[32];
  if (val == invalid)
    strcpy(sz, "*******");
  else
    sprintf(sz, "%ld", val);
  sz[len] = 0;
  for (int i = strlen(sz); i < len; ++i)
    sz[i] = ' ';
  if (len > 0)
    sz[len - 1] = ' ';
  Serial.print(sz);
}

#endif



