/*
   915 MHz Wireless remote data telemetry - Local unit

   by Glen Popiel KW5GP

   uses Adafruit Feather 32U4 LoRa

   uses Adafruit ST7735 and GFX libraries, www.adafruit.com
   uses Radiohead library by Mike McCauley, mikem@airspayce.com

*/

#define debug_mode 1 // Set debug_mode to 1 for Serial Monitor

#include <SPI.h>  // Build-in SPI library
#include <RH_RF95.h>  // Wirehead RF95 library
#include <Adafruit_GFX.h>    // Core graphics library
#include <Adafruit_ST7735.h> // Hardware-specific library

// Define the Feather 32U4 pins
#define RFM95_CS 8
#define RFM95_RST 4
#define RFM95_INT 7

// Define the ST7735 TFT Display pins
#define TFT_CS     10
#define TFT_RST    5
#define TFT_DC     6
#define Backlight 11

// Simplify the TFT display color definitions
#define cyan  ST7735_CYAN
#define yellow  ST7735_YELLOW
#define black  ST7735_BLACK
#define green  ST7735_GREEN
#define white ST7735_WHITE
#define red ST7735_RED

#define LED 13  // Define the onboard LED
#define pushbutton_pin 2 // Define the Alarm Reset pin (Digital pin 2 = Interrupt 1 = SDA pin on Feather 32U4)

int status_color = green; // Variable used to store the display color
float humidity = 0; // Variable to hold the relative humidity
float last_humidity = 0;  // Variable to hold the previous humidity
float temperature = 0;  // Variable to hold the temperature
float last_temperature = 0; // Variable to hold the previous temperature
float voltage = 0;  // Variable to hold the voltage
float last_voltage = 0; // Variable to hold the previous voltage
boolean intrusion = false;  // Flag to indicate an intrusion alarm has been received
boolean data_changed = false;  // Flag to indicate telemetry data has changed
boolean pushbutton = false;  // Flag to indicate alarm reset pushbutton has been pressed
String RX_data; // String to hold the received message
String x; // Temporary variable used to extract data from the received message

uint8_t buf[RH_RF95_MAX_MESSAGE_LEN]; // Receive data buffer - set to RF95 Maximum Message Length
uint8_t len = sizeof(buf); // Variable to hold the size of the Rx buffer

String title_line = "  KW5GP Remote Telemetry"; // Displays at the top of the TFT display

// Define the radio frequency - must match remote's freq!
#define RF95_FREQ 915.0

// Create an instance of the radio driver
RH_RF95 rf95(RFM95_CS, RFM95_INT);

// Create an instance of the TFT display
Adafruit_ST7735 tft = Adafruit_ST7735(TFT_CS, TFT_DC, TFT_RST);

void setup()
{
  pinMode(LED, OUTPUT); // Set the onboard LED pin mode
  pinMode(pushbutton_pin, INPUT_PULLUP);  // Set the alarm reset pushbutton switch pin mode
  pinMode(RFM95_RST, OUTPUT); // Set the Pin Mode for the RF95's Reset pin

  digitalWrite(RFM95_RST, HIGH);  // Turn off RF95 Reset

  if (debug_mode)
  {
    while (!Serial);
    Serial.begin(9600);
    delay(100);
    Serial.println("Feather LoRa RX Test!");
  }

  // Manually reset the RF95 radio
  digitalWrite(RFM95_RST, LOW);
  delay(10);
  digitalWrite(RFM95_RST, HIGH);
  delay(10);

  // Initialize the RF95 radio
  // Defaults after init are 434.0MHz, modulation GFSK_Rb250Fd250, +13dBm, Bw = 125 kHz, Cr = 4/5, Sf = 128chips/symbol, CRC on
  while (!rf95.init())
  {
    if (debug_mode)
    {
      Serial.println("LoRa radio init failed");
    }
    while (1);
  }
  if (debug_mode)
  {
    Serial.println("LoRa radio init OK!");
  }

  // Set the radio frequency
  if (!rf95.setFrequency(RF95_FREQ))
  {
    if (debug_mode)
    {
      Serial.println("setFrequency failed");
    }
    while (1);
  }

  if (debug_mode)
  {
    Serial.print("Set Freq to: "); Serial.println(RF95_FREQ);
  }

  // Set the transmitter power output
  // The default transmitter power is 13dBm, using PA_BOOST.
  // If you are using RFM95/96/97/98 modules which uses the PA_BOOST transmitter pin, then
  // you can set transmitter powers from 5 to 23 dBm:
  rf95.setTxPower(23, false);

  attachInterrupt(digitalPinToInterrupt(pushbutton_pin), read_switch, LOW);  // Set up Interrupt for the Alert Reset pushbutton switch

  pinMode(Backlight, OUTPUT); // Set up the ST7735 TFT Display Backlight pin as output
  digitalWrite(Backlight, HIGH);  // Turn on the TFT Backlight

  // Initialize the ST7735 TFT Color LCD Display
  tft.initR(INITR_BLACKTAB);   // initialize a ST7735S chip, black tab

  tft.setRotation(1); // Rotate the TFT display 90 degrees Clockwise

  tft.fillScreen(black);  // Clear the LCD display

  if (debug_mode)
  {
    Serial.println("TFT Initialized");
  }

}

void loop()
{
  // Check to see if radio is available
  if (rf95.available())
  {
    // Check to see if a message has been received
    if (rf95.recv(buf, &len))
    {
      // Turn on the onboard LED to indicate a message received
      digitalWrite(LED, HIGH);

      RH_RF95::printBuffer("Received: ", buf, len);

      if (debug_mode)
      {
        Serial.print("Got: ");
        Serial.println((char*)buf);
      }

      // Save the receive data to a String
      RX_data = ((char*)buf);
      if (debug_mode)
      {
        Serial.println(RX_data);
        Serial.print("RSSI: ");
        Serial.println(rf95.lastRssi(), DEC);
      }

      unpack(); // Extract the telemetry data from the received message

      if (data_changed) // Only update the display if received data has changed
      {
        update_display(); // Update the LCD display
      }
    } else {
      if (debug_mode)
      {
        Serial.println("Receive failed");
      }
    }
  }
}

// Function to extract the telemetry data from the received message
void unpack()
{
  // Unpack the received packet if it's a valid packet ( First character = 'R')
  if (RX_data[0] == 'R')  // It's a valid message
  {
    x = RX_data.substring(1, 9);  // Extract the humidity string from the message
    if (debug_mode)
    {
      Serial.print(x);
    }
    if (x.substring(5, 8) == "NAN") // If data is "NAN" it means there was an invalid read from the DHT11 sensor
    {
      x = "999";  // Set the display value to "999"
    }
    humidity = x.toFloat(); // Convert the humidity string to a float variable

    if (debug_mode)
    {
      Serial.print("   Humidity: ");
      Serial.println(humidity);
    }

    x = RX_data.substring(10, 17);  // Extract the temperature string from the message

    if (debug_mode)
    {
      Serial.print(x);
    }

    if (x.substring(4, 7) == "NAN") // If data is "NAN" it means there was an invalid read from the DHT11 sensor
    {
      x = "999";  // Set the display value to "999"
    }
    temperature = x.toFloat(); // Convert the temperature string to a float variable

    if (debug_mode)
    {
      Serial.print("   Temp: ");
      Serial.println(temperature);
    }

    x = RX_data.substring(18, 25);  // Extract the voltage string from the message

    if (debug_mode)
    {
      Serial.print(x);
    }

    voltage = x.toFloat(); // Convert the voltage string to a float variable

    if (debug_mode)
    {
      Serial.print("   Volts: ");
      Serial.println(voltage);
    }
  }

  if (debug_mode)
  {
    Serial.print(RX_data[25]);
  }

  // Check the Intrusion string - T = Intrusion Detected, F = No Intrusion Detected
  if (RX_data[25] == 'T') // Check the Intrusion string
  {
    if (debug_mode)
    {
      Serial.println("   Intrusion Detected");
    }
    intrusion = true;
  }

  if (RX_data[25] == 'F')
  {
    if (debug_mode)
    {
      Serial.println("   No Intrusion Detected");
    }
  }

  // Set the data_changed flag if any telemetry data has changed
  if ((humidity != last_humidity) || (temperature != last_temperature) || (voltage != last_voltage) || (intrusion))
  {
    data_changed = true;
  }
}

// Interrupt Function to read the Alert Reset pushbutton switch
void read_switch()
{
  if (!pushbutton) // No need to set flag again if pushbutton flag is set
  {
    if (debug_mode)
    {
      Serial.println("Pushbutton Interrupt detected");
    }
    intrusion = false;  // Clear the Intrusion alert flag
    data_changed = true;  // Set the data changed flag
    pushbutton = true;  // Set the pushbutton switch pressed flag
    update_display(); // Update the display
  }
}

// Funtion to update the TFT Display
void update_display()
{
  tft.fillScreen(black);  // Clear the LCD display
  tft.setTextColor(status_color); // Set the status text color

  // Set the display color to red if there is an intrusion - green otherwise
  if (intrusion)
  {
    status_color = red;
  } else {
    status_color = green;
  }

  tft.setTextColor(status_color); // Set the status text color
  tft.setTextSize(0); // Set the Text size to the smallest font
  tft.setCursor(0, 0);  // Set the display cursor to 0,0
  tft.print(title_line);  // Print the title line
  tft.setTextSize(2); // Set the Text size to a larger font

  tft.setCursor(0, 20);  // Set the cursor position for temperature
  tft.print("Temp:  "); // Display the temperature
  tft.print(temperature, 1);

  tft.setCursor(0, 40);  // Set the cursor position for humidity
  tft.print("R/H:   "); // Display the humidity
  tft.print(humidity, 0);
  tft.print("%");

  tft.setCursor(0, 60);  // Set the cursor position for voltage
  tft.print("Volts: "); // Display the voltage
  tft.print(voltage, 1);

  // Display the Intrusion status
  if (intrusion)
  {
    tft.setCursor(0, 100);
    tft.setTextSize(2); // If there is an alarm, make the text bigger so it stands out
    tft.print("  Intrusion");
    intrusion = true; // Set the Intrusion alarm flag
  } else {
    tft.setTextSize(0);
    tft.setCursor(20, 120);
    tft.print("No Intrusion Detected");
  }

  // Update the previous value variables
  last_temperature  = temperature;
  last_humidity = humidity;
  last_voltage = voltage;

  pushbutton = false; // Clear the alarm reset pushbutton flag
  data_changed = false; // Clear the data changed flag
}



