/*
KW5GP Voice Memory Keyer

220 Memory Voice Memory Keyer using WT588D-U module

*/

#define debug_mode 1  // Enables Debug Mode - Sends Debug output to Serial Monitor

// Include the WT588D Library
#include "WT588D.h"

// Include the Nokia 5110 Library // Nokia 5110 Library
#include <LCD5110_Basic.h>

// Include the Rotary Encoder Library
#include <Encoder.h>

// Set Pin Numbers for the Nokia 5110 LCD Display
const int CLK = 12;
const int DIN = 11;
const int DC = 10;
const int RST = 8;
const int CE = 9;

// set Pin Numbers for the WT588D module
#define WT588D_RST 5  //Module pin "REST" or pin # 1
#define WT588D_CS 3   //Module pin "P02" or pin # 11
#define WT588D_SCL 4  //Module pin "P03" or pin # 10
#define WT588D_SDA 2  //Module pin "P01" or pin # 12
#define WT588D_BUSY 6 //Module pin "LED/BUSY" or pin # 15

// Set the Pin Numbers for the Rotary Encoder
const int encoder_PinA = 14;  // Pin A of the Rotary Encoder (A0 Pin)
const int encoder_PinB = 15;  // Pin B of the Rotary Encoder (A1 Pin)
const int Pushbutton_Pin = 7;  // Pushbutton Switch input Pin
const int LED_Pin = 16; // The LED Pin
const int TX_Key_Pin = 13;  // The Transmit Relay Pin

const int debounceInterval = 500;  //debounce for 500 ms between pushbutton press

int encoder_Pos = 0; // The current encoder position
int old_encoder_Pos = 0; // The previous encoder position
boolean Pushbutton_Switch;  // Variable to read the state of the pushbutton switch
boolean switch_pressed = false; // flag to indicate the pushbutton switch has been pressed

int memory_select = 0; // The starting memory select value
int range_high = 219; // The max setting for the memory select option
int range_low = 0; // The minimum setting for the memory select option

unsigned long int delay_time = 1000; // pushbutton delay time between switch presses

// Instantiate the LCD
LCD5110 glcd(CLK,DIN,DC,RST,CE); //  Assign the Nokia 5110 LCD Pins

extern uint8_t SmallFont[];  // define the Nokia Font  

// Instantiate the Rotary Encoder
Encoder Enc(encoder_PinA, encoder_PinB);  // Assign the Encoder Pins

// Instantiate the WT588D
WT588D myWT588D(WT588D_RST, WT588D_CS, WT588D_SCL, WT588D_SDA, WT588D_BUSY);  // Assign the WT588D Pins

void setup() 
{
  if (debug_mode) // Enable the Serial Monitor port and set to 9600 baud
  {
    Serial.begin(9600);
  }

  // initialize the WT588D chip and port mapping
  myWT588D.begin();

  glcd.InitLCD(60);  // Initialize the Nokia LCD and set contrast value
  glcd.setFont(SmallFont);  // Use Small Font
  
  // Display the Startup screen
  glcd.clrScr();
  glcd.print("KW5GP", CENTER,0);
  glcd.print("Voice Memory",CENTER,16);
  glcd.print("Initializing", CENTER,40);
  
  delay(3000);
  
  glcd.clrScr();  //Clear the LCD screen

  // Setup the LCD Display for operation
  glcd.print("Ready",CENTER,0);  // Display the Information template on the LCD
  glcd.print("Memory:",0,16);  // Display the Information template on the LCD  
  glcd.printNumI(memory_select,40,16,3,' ');  
  glcd.print("Idle",CENTER,32);

 // Set up the Rotary Encoder and Pushbutton switch and enable the Internal Pull-up resistor on the Inputs 
  pinMode (encoder_PinA,INPUT_PULLUP);
  pinMode (encoder_PinB,INPUT_PULLUP);
  pinMode (Pushbutton_Pin, INPUT_PULLUP);
  pinMode (LED_Pin, OUTPUT);  // Setup the LED pin for output
  pinMode (TX_Key_Pin, OUTPUT); // Setup the TX Relay pin for output
  
}

void loop() 
{
  read_encoder(); // Read the rotary encoder position and update selection and display if changed
  
  read_pushbutton();  // Check to see if the pushbutton switch has been pressed

  if(switch_pressed)  // The switch has been pressed - send message
  {
    glcd.print("Sending: ",0,32); // Update the LCD and display what message we are sending
    glcd.printNumI(memory_select,52,32); 
    digitalWrite(LED_Pin, HIGH);  // Turn on the LED
    digitalWrite(TX_Key_Pin, HIGH); // Turn on the TX relay
    playMessage(memory_select); // Play the selected message
    digitalWrite(LED_Pin, LOW); // Turn off the LED
    digitalWrite(TX_Key_Pin, LOW);  // Turn off the TX relay
    switch_pressed = false; // Reset the switch pressed flag
    glcd.clrRow(4); // Update the LCD display
    glcd.print("Idle",CENTER,32);    
  }

}

// Function to read the rotary encoder
void read_encoder()
{
  // Read the Encoder  
  encoder_Pos = Enc.read()/4; // divide by 4 to match encoder detent
  if (encoder_Pos != old_encoder_Pos) // If the Encoder has changed update selection 
  {
    if (encoder_Pos > old_encoder_Pos) // If we're increasing count
    {
      if (memory_select >= range_high) // Limit to top end of count
      {
        memory_select = range_high;
      } else {
        memory_select = memory_select + 1 ;
      }
     
    } else {
      if (memory_select <= range_low) // Limit to the low end of count
      {
        memory_select = range_low;
      } else {
        memory_select = memory_select - 1;
      }
    }

    old_encoder_Pos = encoder_Pos;  // Set the previous encoder position to the current position
    
    glcd.printNumI(memory_select,40,16,3,' ');  // Update the LCD display
  }
}  

// Function to read the pushbutton switch
void read_pushbutton()
{
  // Read the Encoder Pushbutton Switch
  Pushbutton_Switch = digitalRead(Pushbutton_Pin);
  if(Pushbutton_Switch == LOW && millis() > delay_time) // Check to see if pressed
  {
    // if it's changed, trigger the message but don't allow again until debounce time has passed
    delay_time = millis() + debounceInterval; // if it's changed, set the switch pressed flag but don't allow again for debounce time
    switch_pressed = true;
  }
}  

// Function to play the selected message
void playMessage(int messageNumber)
{
  myWT588D.playSound(messageNumber);  // Play the selected message

  delay(100); //give the WT588D module time to start sending

  if (debug_mode) // Tell us which message is playing if in debug mode
  {
    Serial.print("Playing: ");
    Serial.println(messageNumber);
  }
    
  while (myWT588D.isBusy())  // Wait while the WT588D is busy sending the message
  {
    // Do nothing while we wait for the message to finish
  }

  if (debug_mode) // Tell us when the message has ended if in debug mode
  {
    Serial.println("Message Sent");
  }
}
