/*
KW5GP Load Tester

Switches in a variable load for testing 12V Batteries and Power Supplies

*/

#define debug_mode 0  // Enables Debug Mode - Sends Debug output to Serial Monitor

// Include Timer Library so we can do timed interrupts
#include "Timer.h"  

// Include the Nokia 5110 Library // Nokia 5110 Library
#include <LCD5110_Basic.h>

// Include the Rotary Encoder Library
#include <Encoder.h>

#define voltage_pin A0  // Define the input voltage analog pin
int relay[] = {0,5,6,7,15,16,17,18}; // Define the Output Pins for the relays. Note: Pins 15-18 are the designation for A1-A4

// Set Pin Numbers for the Nokia 5110 LCD Display
const int CLK = 12;
const int DIN = 11;
const int DC = 10;
const int RST = 8;
const int CE = 9;

// Set the Pin Numbers for the Rotary Encoder

const int encoder_PinA = 2;  // Pin A of the Rotary Encoder
const int encoder_PinB = 3;  // Pin B of the Rotary Encoder
const int encoder_Switch_Pin = 4;  // Encoder Pushbutton Switch input pin

const int debounceInterval = 500;  //disable step change for 500 ms between encoder pushbutton press

boolean encoder_Switch; // Encoder Pushbutton Switch Press flag
boolean relay_enable = false; // Flag to enable the relays to energize
boolean state_change = true; // Flag to indicate a change from Inactive to Active or vice versa
int encoder_Pos = 0; // The current encoder position
int old_encoder_Pos = 0; // The previous encoder position
int relay_select = 0; // The relay (load) select value
int range_high = 8; // The max setting for the relay select option
int range_low = 0; // The minimum setting for the relay select option
int resistors = 0; // number of resistors to enable
float voltage;  // Variable to store the voltage
float amps; // Variable to store the amperage
float load_ohms; // The current load in ohms
float volt_drop = .89;  // The voltage drop across the 1N4001 diode and path to the analog input
int volt_cal = 2450;  // Voltage calibration value for the MAP statement
unsigned long int delay_time = 2000; // Encoder pushbutton debounce delay time
String selected = "None"; // The current text for the encoder selection
String old_selected =" "; // The previous text for the encoder selection
int decimals = 1;
int elapsed_minutes, elapsed_hours, elapsed_seconds;

boolean timing = false;  // Variable to let us know that timing has started
long seconds = 0;  // The number of seconds since the timer started
long last_second = 0; // Store the value of the previous second
int tickEvent;  // The process number for the one second Timer tick

// Instantiate the LCD
LCD5110 glcd(CLK,DIN,DC,RST,CE); //  Assign the Nokia 5110 LCD Pins

extern uint8_t SmallFont[];  // define the Nokia Font  

// Instantiate the Rotary Encoder
Encoder Enc(encoder_PinA, encoder_PinB);

Timer t;  // Create Timer object as t

void setup() 
{
  if (debug_mode) // Enable the Serial Monitor port and set to 9600 baud
  {
    Serial.begin(9600);
  }

  glcd.InitLCD(60);  // Initialize the Nokia LCD and set contrast value
  glcd.setFont(SmallFont);  // Use Small Font
  
  // Display the Startup screen
  glcd.clrScr();
  glcd.print("KW5GP", CENTER,0);
  glcd.print("Load Tester",CENTER,16);
  glcd.print("Initializing", CENTER,40);
  
  // initialize relay output pins - uses an array to set the pin mode for the relays and sets the pins to OUTPUT mode
  for (int x = 1; x <=7; x++)
  {
    pinMode(relay[x], OUTPUT);
    if (debug_mode)
    {
      Serial.print("Relay pin set for output: ");
      Serial.println(relay[x]);
    }
  }
   
  tickEvent = t.every(1000, Tick); // Set up the one second timer interrupt

  if (debug_mode) // Start a one second Timer tick for debug
  {
    Serial.print("1 second tick started Process id= ");
    Serial.println(tickEvent);
  }  

  delay(3000);
  
  glcd.clrScr();  //Clear the LCD screen

  // Setup the LCD Display for operation
  glcd.print("Volts:",0,0);  // Display the Information template on the LCD
  glcd.print("Load :",0,8);
  glcd.print("Amps :",0,16);
  glcd.print("Ohms :",0,24);
  glcd.print("State:",0,32);
  glcd.print("Time : ",0,40);
  
 // Set up the Rotary Encoder and enable the Internal Pull-up resistor on the Encoder Inputs 
  pinMode (encoder_PinA,INPUT_PULLUP);
  pinMode (encoder_PinB,INPUT_PULLUP);
  pinMode (encoder_Switch_Pin, INPUT_PULLUP);
  
}

void loop() 
{
  if (!relay_enable)  // Check that the relays are not enabled
  {
    // Don't read or change the relays selected if load is enabled
    read_encoder(); // Read the encoder to select load
  }
  read_pushbutton();  // Read the Encoder Pushbutton switch to enable load

  glcd.printNumF(voltage,1,50,0); // Display the load voltage
  
  if (relay_enable && relay_select != 0)
  {
    glcd.printNumF(load_ohms,decimals,50,8);  // Display the load value when relays are enabled
  } else {
    glcd.print("Off    ",50,8);
  }
    
  glcd.printNumF(amps,1,50,16); // Display the amperage calculated by load resistor value and load value
  
  switch (relay_select) // Display the selected load value in ohms based on rotary encoder selection
  {
    case 0:
    selected = "None";
    break;

    case 1:
    selected = "8.0";
    break;

    case 2:
    selected = "4.0 ";
    break;

    case 3:
    selected = "2.0";
    break;

    case 4:
    selected = "1.333";
    break;

    case 5:
    selected = "1.0";
    break;

    case 6:
    selected = "0.8";
    break;

    case 7:
    selected = "0.667";
    break;

    case 8:
    selected = "0.615";
    break;
    
  }

  if (selected != old_selected) // If the rotary encoder value has changed, update the LCD
  {
    glcd.clrRow(3);
    glcd.print("Ohms :",0,24);    
    glcd.print(selected,50,30);
    old_selected = selected;
  }
  
  if (state_change) // If the rotary encoder pushbutton is pressed enable/disable relays
  {
    glcd.clrRow(4);
    glcd.print("State:",0,32);    
    if (relay_enable)
    {
      glcd.print("Active",40,32);
    } else 
    {
      glcd.print("No Load",40,32);
    }
    state_change = false;
  }

  if (seconds != last_second) // Update and the time if it has changed
  {
    //convert seconds to HH:MM:SS
    elapsed_seconds = seconds % 60;
    elapsed_minutes = (seconds/60) % 60;
    elapsed_hours = (seconds/3600) % 3600;
    if (debug_mode)
    {
      Serial.print("Converted Time: ");
      Serial.print(elapsed_hours);
      Serial.print(" : ");
      Serial.print(elapsed_minutes);
      Serial.print(" : ");
      Serial.println(elapsed_seconds);
    }
  
    glcd.printNumI(elapsed_hours,36,40,2,'0');
    glcd.print(":",48,40);
    glcd.printNumI(elapsed_minutes,54,40,2,'0');
    glcd.print(":",66,40);
    glcd.printNumI(elapsed_seconds,72,40,2,'0');
    last_second = seconds;
  }

  t.update();  // Update the Interrupt handler - required for the Timer library to function 

}

// Function to provide a one second Timer tick
void Tick() 
{
  readVoltage();  // Read the input voltage

  if (relay_enable) // If the relays are enabled, update the elapsed time
  {
    seconds = seconds +1 ;  // increment the seconds counter
    if (relay_select != 0)
    {
      amps = voltage / load_ohms; // Update the calculated amperage value
    } else {
      amps = 0;
    }
  }

  if (debug_mode)
  {
    Serial.print ("Seconds: "); // Print the Seconds and voltage
    Serial.print (seconds);
    Serial.print ("   Volts: ");
    Serial.println(voltage);
  }
}

// Function to read the input voltage
void readVoltage()
{
  float count = analogRead(voltage_pin);
  voltage = map(count,0,1023,0,volt_cal); // Map the A/D value to a voltage in millivolts
  voltage = (voltage/100)+ volt_drop;// Convert to volts and subtract the input circuit voltage loss
}

// Function to read the rotary encoder
void read_encoder()
{
  // Read the Encoder  
  encoder_Pos = Enc.read()/4; // divide by 4 to match encoder detent
  if (encoder_Pos != old_encoder_Pos) // If the Encoder has changed update selection 
  {
    if (encoder_Pos > old_encoder_Pos) // If we're increasing count
    {
      if (relay_select >= range_high) // Limit to top end of count
      {
        relay_select = range_high;
      } else {
        relay_select = relay_select + 1 ;
      }
     
    } else {
      if (relay_select <= range_low)
      {
        relay_select = range_low;
      } else {
        relay_select = relay_select - 1;
      }
    }

    old_encoder_Pos = encoder_Pos;  // Set the previous encoder position to the current position
   
  }
}  

// Function to read the rotary encoder pushbutton switch
void read_pushbutton()
{
  // Read the Encoder Pushbutton Switch
  encoder_Switch = digitalRead(encoder_Switch_Pin);
  if(encoder_Switch == LOW && millis() > delay_time) // Check to see if pressed
  {
    // if it's changed, toggle the step size but don't allow again until debounce time has passed
    delay_time = millis() + debounceInterval; // if it's changed, toggle the step size but don't allow again for debounce time
    state_change = true;

    if (relay_enable)
    {
      // Turn the relays off
      relays_off();
    } else {
      // Turn the relays on
      relays_on();
    }
  }
}  

// Function to turn all relays off
void relays_off()
{
  relay_enable = false;
  for (int x = 1; x <=8; x++)
  {
    digitalWrite(relay[x], LOW);
  }
  amps = 0;
}

// Function to figure out which relays we need to turn on
void relays_on()
{
  relay_enable = true;  // Indicates that relays are enabled
  seconds = 0;  

  switch (relay_select)
  {
    case 0:
    // Don't turn on any relays, just start the timer
    break;

    case 1:
    // Set 8 Ohm load - 1.5A@12V
    load_ohms = 8;
    decimals = 1;
    digitalWrite(relay[1], HIGH);
    if (debug_mode)
    {
      Serial.println("Setting resistor relay: 1");
    } 
  
    break;

    case 2:
    // Set 4 Ohm load - 3A@12V
    load_ohms = 4;
    decimals = 1;    
    digitalWrite(relay[2], HIGH);
    if (debug_mode)
    {
      Serial.println("Setting resistor relay: 2");
    }    
   break;

    case 3:
    // set 2 Ohm load - 6A@12V
    load_ohms = 2;
    decimals = 1;
    resistors = 2;
    set_load();
    break;

    case 4:
    // set 1.3 Ohm load - 9A@12V
    load_ohms = 1.333;
    decimals = 3;
    resistors = 3;
    set_load();
    break;
    
    case 5:
    // set 1 Ohm load - 12A@12V
    load_ohms = 1;
    decimals = 1;    
    resistors = 4;
    set_load();
    break;

    case 6:
    // set .8 Ohm load - 15A@12V
    load_ohms = .8;
    decimals = 1;    
    resistors = 5;
    set_load();
    break;

    case 7:
    // set .67 Ohm load - 18A@12V
    load_ohms = .667;
    decimals = 3;    
    resistors = 6;
    set_load();
    break;

    case 8:
    // set .615 Ohm load - 19.5A@12V - all resistors on
    load_ohms = .615;
    decimals = 3;    
    digitalWrite(relay[1], HIGH);
    if (debug_mode)
    {
      Serial.println("Setting resistor relay: 1");
    }
    resistors = 6;
    set_load();
    break;
    
    
  }
  
}

// Function to set the load resistor relays
void set_load()
{
  if (debug_mode)
  {
    Serial.print("Relay Select: ");
    Serial.print(relay_select);
    Serial.print("     Load Ohms: ");
    Serial.println(load_ohms,decimals);
  }
  for (int x = 2; x <= (resistors + 1); x++)
  {
    digitalWrite(relay[x], HIGH);
    if (debug_mode)
    {
      Serial.print("Setting resistor relay: ");
      Serial.println(x);
    }
  }
}

